﻿// Copyright (c) 2010, Object Computing, Inc.
// All rights reserved.

using System;
using System.Collections.Generic;
using System.Text;
using System.Runtime.InteropServices;
using System.Diagnostics;

class ZmqDotnetPublisher
{

    static unsafe int Main(string[] args)
    {
        if (args.Length < 4)
        {
           Console.Out.WriteLine("usage: ZmqDotnetPublisher <message-size> <roundtrip-count> " +
                "<bind-to-sub> [<connect-to-pub>]+\n");
           Console.Out.WriteLine(" e.g.: ZmqDotnetPublisher 1000 10000 tcp://10.201.200.72:54321 tcp://10.201.200.72:54322\n");
           return 1;
        }

        uint messageSize = Convert.ToUInt32(args[0]);
        int roundtripCount = Convert.ToInt32(args[1]);
        String bind_to_sub = args[2];

        //  Initialise 0MQ infrastructure
        ZMQ.Context ctx = new ZMQ.Context(1, 1, 0);
        ZMQ.Socket pub = ctx.Socket(ZMQ.PUB);
        ZMQ.Socket sub = ctx.Socket(ZMQ.SUB);
        sub.SetSockOpt(ZMQ.SUBSCRIBE, "DataFeed");

        Console.Out.WriteLine("Binding to " + bind_to_sub);
        sub.Bind(bind_to_sub);

        int num_subscribers = 0;
        for (int i = 3; i < args.Length; ++i)
        {
           Console.Out.WriteLine("Connecting to " + args[i]);
           pub.Connect(args[i]);
           ++num_subscribers;
        }

        //  Create a message to send.
        byte[] msg = new byte[messageSize];
        new System.Text.ASCIIEncoding().GetBytes("DataFeed\x00").CopyTo(msg,0);

        Console.Out.WriteLine("Sending messages -- " + num_subscribers + " subscribers, "
           + roundtripCount + " messages, size = " + messageSize);

        //  Start measuring the time.
        System.Diagnostics.Stopwatch watch;
        watch = new Stopwatch();
        watch.Start();

        //  Start sending messages.
        for (int i = 0; i < roundtripCount; i++)
        {
            pub.Send(msg);
            for (int jj = 0; jj < num_subscribers; ++jj)
            {
               sub.Recv(out msg);
               Debug.Assert(msg.Length == messageSize);
            }
        }

        //  Stop measuring the time.
        watch.Stop();
        Int64 elapsedTime = watch.ElapsedTicks;

        //  Print out the test parameters.
        Console.Out.WriteLine("message size: " + messageSize + " [B]");
        Console.Out.WriteLine("roundtrip count: " + roundtripCount);

        //  Compute and print out the latency.
        double latency = (double)(elapsedTime) / roundtripCount / 2 *
            1000000 / Stopwatch.Frequency / (double)num_subscribers;
        Console.Out.WriteLine("\n\nYour average latency is {0} [us]\n\n",
            latency.ToString("f2"));

        return 0;
    }
}
