// Copyright (c) 2010, Object Computing, Inc.
// All rights reserved.

#include "DdsUtilPch.h"
#include "DoneTokenManager.h"

// OpenDDS
#include "RawTypeSupportImpl.h"
#include <dds/DCPS/Service_Participant.h>
#include <dds/DCPS/Marked_Default_Qos.h>

#include <iostream>

// Profiling
#include "Profiler.h"

namespace MiddlewareNewsBrief {

DoneTokenManager::DoneTokenManager()
: isDone_(false)
{
}

DoneTokenManager::~DoneTokenManager()
{
}

void
DoneTokenManager::initWriter(DDS::DomainParticipant_ptr participant,
                             DDS::Publisher_ptr publisher,
                             DDS::Subscriber_ptr subscriber)
{
  this->is_original_publisher_side_ = true;
  
  // Create and register the DoneToken type support object
  MiddlewareNewsBrief::DoneTokenTypeSupport_var ts = 
    new MiddlewareNewsBrief::DoneTokenTypeSupportImpl();
  if (DDS::RETCODE_OK != ts->register_type(
                           participant,
                           MiddlewareNewsBrief::DONE_TOKEN_TOPIC_TYPE)) 
  {
    std::ostringstream msg;
    msg << "register_type for " << MiddlewareNewsBrief::DONE_TOKEN_TOPIC_TYPE 
        << " failed." << std::ends;
    throw std::runtime_error(msg.str());
  }

  // Get the default QoS for the two topics
  DDS::TopicQos topic_qos;
  participant->get_default_topic_qos(topic_qos);

  // Create a topic for the SecurityTopic type...
  DDS::Topic_var topic =
    participant->create_topic (MiddlewareNewsBrief::DONE_TOKEN_TOPIC_NAME,
                               MiddlewareNewsBrief::DONE_TOKEN_TOPIC_TYPE,
                               topic_qos,
                               0,
                               0);
  if (topic == 0) 
  {
    std::ostringstream msg;
    msg << "create_topic for " << MiddlewareNewsBrief::DONE_TOKEN_TOPIC_TYPE 
        << " failed." << std::ends;
    throw std::runtime_error(msg.str());
  }

  // Get QoS to use for our DataWriter
  DDS::DataWriterQos dw_qos;
  publisher->get_default_datawriter_qos (dw_qos);
  dw_qos.reliability.kind = DDS::RELIABLE_RELIABILITY_QOS;
  dw_qos.writer_data_lifecycle.autodispose_unregistered_instances = false;

  // Create a DataWriter
  DDS::DataWriter_var dw = 
    publisher->create_datawriter(topic.in (),
                                 dw_qos,
                                 0,
                                 0);
  if (dw == 0) 
  {
    std::ostringstream msg;
    msg << "create_datawriter for " 
        << MiddlewareNewsBrief::DONE_TOKEN_TOPIC_NAME << " failed." 
        << std::ends;
    throw std::runtime_error(msg.str());
  }

  this->writer_ = MiddlewareNewsBrief::DoneTokenDataWriter::_narrow(dw.in());
  if (this->writer_ == 0) 
  {
    std::ostringstream msg;
    msg << "narrow for " 
        << MiddlewareNewsBrief::DONE_TOKEN_TOPIC_NAME << " failed." 
        << std::ends;
    throw std::runtime_error(msg.str());
  }

  // Get QoS to use for our DataReader
  DDS::DataReaderQos dr_qos;
  subscriber->get_default_datareader_qos (dr_qos);
  dr_qos.reliability.kind = DDS::RELIABLE_RELIABILITY_QOS;

  // Create a DataReader
  DDS::DataReader_var echo_topic_data_reader = 
    subscriber->create_datareader(topic.in (),
                                  dr_qos,
                                  this,
                                  DDS::DATA_AVAILABLE_STATUS);
  if (echo_topic_data_reader == 0) 
  {
    std::ostringstream msg;
    msg << "create_datareader for " 
        << MiddlewareNewsBrief::DONE_TOKEN_TOPIC_TYPE << " failed." 
        << std::ends;
    throw std::runtime_error(msg.str());
  }
}


void
DoneTokenManager::writeDone(const std::string& message)
{
  if (this->writer_ == 0)
  {
    throw new std::runtime_error("DoneTokenManager::writeDone needs a valid "
                                 "writer; call initWriter first");
  }

  MiddlewareNewsBrief::DoneToken sample;
  sample.message = message.c_str();
  DDS::ReturnCode_t retcode = this->writer_->write(sample,DDS::HANDLE_NIL);
  if (retcode != DDS::RETCODE_OK)
  {
    std::ostringstream msg;
    msg << "DoneTokenManager::writeDone failed with code " << retcode
        << std::ends;
    throw new std::runtime_error(msg.str());
  }
}

void
DoneTokenManager::initReader(DDS::DomainParticipant_ptr participant,
                             DDS::Publisher_ptr publisher,
                             DDS::Subscriber_ptr subscriber)
{
  this->is_original_publisher_side_ = false;

  // Create and register the DoneToken type support object
  MiddlewareNewsBrief::DoneTokenTypeSupport_var ts = 
    new MiddlewareNewsBrief::DoneTokenTypeSupportImpl();
  if (DDS::RETCODE_OK != ts->register_type(
                           participant,
                           MiddlewareNewsBrief::DONE_TOKEN_TOPIC_TYPE)) 
  {
    std::ostringstream msg;
    msg << "register_type for " << MiddlewareNewsBrief::DONE_TOKEN_TOPIC_TYPE 
        << " failed." << std::ends;
    throw std::runtime_error(msg.str());
  }

  // Get the default QoS for the two topics
  DDS::TopicQos topic_qos;
  participant->get_default_topic_qos(topic_qos);

  // Create a topic for the SecurityTopic type...
  DDS::Topic_var topic =
    participant->create_topic (MiddlewareNewsBrief::DONE_TOKEN_TOPIC_NAME,
                               MiddlewareNewsBrief::DONE_TOKEN_TOPIC_TYPE,
                               topic_qos,
                               0,
                               0);
  if (topic == 0) 
  {
    std::ostringstream msg;
    msg << "create_topic for " << MiddlewareNewsBrief::DONE_TOKEN_TOPIC_TYPE 
        << " failed." << std::ends;
    throw std::runtime_error(msg.str());
  }

  // Get QoS to use for our DataReader
  DDS::DataReaderQos dr_qos;
  subscriber->get_default_datareader_qos (dr_qos);
  dr_qos.reliability.kind = DDS::RELIABLE_RELIABILITY_QOS;

  // Create a DataReader
  DDS::DataReader_var security_topic_data_reader = 
    subscriber->create_datareader(topic.in (),
                                  dr_qos,
                                  this,
                                  DDS::DATA_AVAILABLE_STATUS);
  if (security_topic_data_reader == 0) 
  {
    std::ostringstream msg;
    msg << "create_datareader for " 
        << MiddlewareNewsBrief::DONE_TOKEN_TOPIC_TYPE << " failed." 
        << std::ends;
    throw std::runtime_error(msg.str());
  }

  // Get QoS to use for our DataWriter
  DDS::DataWriterQos dw_qos;
  publisher->get_default_datawriter_qos (dw_qos);
  dw_qos.reliability.kind = DDS::RELIABLE_RELIABILITY_QOS;
  dw_qos.writer_data_lifecycle.autodispose_unregistered_instances = false;

  // Create a DataWriter
  DDS::DataWriter_var dw = 
    publisher->create_datawriter(topic.in (),
                                 dw_qos,
                                 0,
                                 0);
  if (dw == 0) 
  {
    std::ostringstream msg;
    msg << "create_datawriter failed" << std::ends;
    throw std::runtime_error(msg.str());
  }

  this->writer_ = MiddlewareNewsBrief::DoneTokenDataWriter::_narrow(dw.in());
  if (this->writer_ == 0) 
  {
    std::ostringstream msg;
    msg << "narrow failed" << std::ends;
    throw std::runtime_error(msg.str());
  }
}

bool
DoneTokenManager::isDone() const
{
  ACE_Guard<ACE_Thread_Mutex> guard(this->lock_);
  return this->isDone_;
}


void
DoneTokenManager::do_echo()
{
  MiddlewareNewsBrief::DoneTokenDataWriter_var w;
  {
    ACE_Guard<ACE_Thread_Mutex> guard(this->lock_);
    if (this->isDone_ 
        && !this->is_original_publisher_side_ 
        && this->writer_ != 0) 
    {
      w = MiddlewareNewsBrief::DoneTokenDataWriter::_duplicate(this->writer_.in());
    }
  }

  if (w != 0) 
  {
    MiddlewareNewsBrief::DoneToken sample;
    sample.message = "Echo";
    
    // Echo the "Done" token back
    w->write(sample,DDS::HANDLE_NIL);
  }
}

void
DoneTokenManager::fini(DDS::Publisher_ptr publisher)
{
  ACE_Guard<ACE_Thread_Mutex> guard(this->lock_);
  if (this->writer_ != 0) {
    publisher->delete_datawriter(this->writer_.in());
    this->writer_ = MiddlewareNewsBrief::DoneTokenDataWriter::_nil();
  }
}

void
DoneTokenManager::on_data_available(DDS::DataReader_ptr reader)
{
  MiddlewareNewsBrief::DoneTokenDataReader_var topic_dr = 
    MiddlewareNewsBrief::DoneTokenDataReader::_narrow(reader);
  if (topic_dr == 0) 
  {
    std::cerr << "MiddlewareNewsBrief::DoneTokenDataReader_var::"
              << "on_data_available: _narrow failed." << std::endl;
    return;
  }

  MiddlewareNewsBrief::DoneToken sample;
  DDS::SampleInfo si;
  DDS::ReturnCode_t status = topic_dr->take_next_sample(sample,si);

  if (status == DDS::RETCODE_OK 
      && si.instance_state == DDS::ALIVE_INSTANCE_STATE)
  {
    ACE_Guard<ACE_Thread_Mutex> guard(this->lock_);
    this->isDone_ = true;
  } 
}


} // MiddlewareNewsBrief


