// Copyright (c) 2009, Object Computing, Inc.
// All rights reserved.
// See the file license.txt for licensing information.
//
#ifdef _MSC_VER
# pragma once
#endif
#ifndef MIDDLEWARENEWSBRIEF_DDSUTIL_ECHO_WRITER_T_H
#define MIDDLEWARENEWSBRIEF_DDSUTIL_ECHO_WRITER_T_H

#include "DDSUtil.h"
#include "Profiler.h"

// OpenDDS
#include <dds/DCPS/Service_Participant.h>
#include <dds/DCPS/PublisherImpl.h>

#include <string>
#include <list>

namespace MiddlewareNewsBrief{

    template <typename DDS_STRUCT_T> struct DDSTypeDetails;
  
    template <typename DDS_STRUCT_T>
    class EchoWriter
      : public virtual OpenDDS::DCPS::LocalObject<DDS::DataReaderListener>
    {
    public:
      typedef DDSTypeDetails<DDS_STRUCT_T> Details;

      EchoWriter();
      virtual ~EchoWriter();

      void init(DDS::DomainParticipant_ptr participant,
                DDS::Publisher_ptr publisher,
                DDS::DataReader_ptr source,
                CORBA::ULong DDS_STRUCT_T::* timestampField,
                DDS_STRUCT_T instancePrototype = DDS_STRUCT_T());

      void do_echo(size_t sleepUsecBetweenWrites = 0);

      virtual void on_data_available(
        DDS::DataReader_ptr reader
      )
      throw (CORBA::SystemException);

      virtual void on_requested_deadline_missed (
        DDS::DataReader_ptr reader,
        const DDS::RequestedDeadlineMissedStatus & status)
        throw (CORBA::SystemException)
      {}

      virtual void on_requested_incompatible_qos (
        DDS::DataReader_ptr reader,
        const DDS::RequestedIncompatibleQosStatus & status)
      throw (CORBA::SystemException)
      {}

      virtual void on_liveliness_changed (
        DDS::DataReader_ptr reader,
        const DDS::LivelinessChangedStatus & status)
      throw (CORBA::SystemException)
      {}

      virtual void on_subscription_matched (
        DDS::DataReader_ptr reader,
        const DDS::SubscriptionMatchedStatus & status
      )
      throw (CORBA::SystemException)
      {}

      virtual void on_sample_rejected(
        DDS::DataReader_ptr reader,
        const DDS::SampleRejectedStatus& status
      )
      throw (CORBA::SystemException)
      {}

      virtual void on_sample_lost(
        DDS::DataReader_ptr reader,
        const DDS::SampleLostStatus& status
      )
      throw (CORBA::SystemException){}

    private:

      // disable copy and assignment
      EchoWriter(const EchoWriter<DDS_STRUCT_T>& rhs);
      void operator=(const EchoWriter<DDS_STRUCT_T>& rhs);
      
      // OpenDDS
      typename Details::Writer::_var_type writer_;
      DDS::InstanceHandle_t instance_handle_;

      CORBA::ULong DDS_STRUCT_T::* timestampField_;
      const std::string profileString_;

      class SampleWrapper {
      public:
        SampleWrapper(const DDS_STRUCT_T& sample,
                      MIDDLEWARENEWSBRIEF_PROFILER_TIME_TYPE creation_time,
                      CORBA::ULong DDS_STRUCT_T::* timestampField)
          : sample_(sample)
          , creation_time_(creation_time)
          , timestampField_(timestampField)
        {
        }

        const DDS_STRUCT_T& getSample() {
          // Calculate time lapsed since this "echo" sample was saved off;
          // subtract that time from the sample's timestamps.  We're just
          // trying to measure round trip latency over OpenDDS, not the
          // effects of backpressure
          MIDDLEWARENEWSBRIEF_PROFILER_TIME_TYPE now =
            MIDDLEWARENEWSBRIEF_PROFILER_GET_TIME;
          MIDDLEWARENEWSBRIEF_PROFILER_TIME_TYPE diff =
            MIDDLEWARENEWSBRIEF_PROFILER_DIFF(now,this->creation_time_);

          if (this->timestampField_ != 0) {
            this->sample_.*(this->timestampField_) += diff;
          }

          return this->sample_;
        }
      private:
        DDS_STRUCT_T sample_;
        const MIDDLEWARENEWSBRIEF_PROFILER_TIME_TYPE creation_time_;
        CORBA::ULong DDS_STRUCT_T::* timestampField_;
      };

      std::list<SampleWrapper> waiting_for_echo_;
    };



template <typename DDS_STRUCT_T>
EchoWriter<DDS_STRUCT_T>::EchoWriter()
  : instance_handle_(DDS::HANDLE_NIL)
  , profileString_("Reader<" + Details::type_name + ">::on_data_avail()")
  , timestampField_(0)
{
}

template <typename DDS_STRUCT_T>
EchoWriter<DDS_STRUCT_T>::~EchoWriter()
{
}

template <typename DDS_STRUCT_T>
void
EchoWriter<DDS_STRUCT_T>::init(
  DDS::DomainParticipant_ptr participant,
  DDS::Publisher_ptr publisher,
  DDS::DataReader_ptr source,
  CORBA::ULong DDS_STRUCT_T::* timestampField,
  DDS_STRUCT_T instancePrototype)
{
  this->timestampField_ = timestampField;

  DDS::TopicDescription_var source_topic_desc = source->get_topicdescription();
  CORBA::String_var topic_name = source_topic_desc->get_name();

  DDS::DataWriter_var dw =
    DDSUtil::create_datawriter<typename Details::TypeSupportImpl>(participant,
                                                                  publisher,
                                                                  topic_name.in());

  // narrow the data writer created above 
  // and make sure the narrow's result is non-nil.
  this->writer_ = Details::Writer::_narrow(dw.in());
  if (this->writer_ == 0) 
  {
    std::ostringstream msg;
    msg  << Details::type_name << "DataWriter could not be narrowed for topic "
         << topic_name.in() << std::ends;
    throw std::runtime_error(msg.str());
  }

  DDS_STRUCT_T sample = instancePrototype;
  this->instance_handle_ = this->writer_->register_instance(sample);

  // Write once to make sure the transport is open
  if (this->timestampField_ != 0) {
    sample.*(this->timestampField_) = 0;
    this->writer_->write(sample,this->instance_handle_);
  }

  // Set this object as the DataReader's listener
  source->set_listener(this,DDS::DATA_AVAILABLE_STATUS);
}


template <typename DDS_STRUCT_T>
void
EchoWriter<DDS_STRUCT_T>::do_echo(size_t sleepUsecBetweenWrites)
{
  for (typename std::list<SampleWrapper>::iterator itr =
         this->waiting_for_echo_.begin();
       itr != this->waiting_for_echo_.end();
       ++itr)
  {
    // Sleep between writes, if desired
    if (sleepUsecBetweenWrites > 0)
    { 
      boost::this_thread::sleep
        (boost::posix_time::microseconds(sleepUsecBetweenWrites));
    }
    
    this->writer_->write(itr->getSample(),this->instance_handle_);    
  }
  this->waiting_for_echo_.clear();
}


template <typename DDS_STRUCT_T>
void
EchoWriter<DDS_STRUCT_T>::on_data_available(DDS::DataReader_ptr reader)
  throw (CORBA::SystemException)
{
  typename Details::Reader::_var_type dr = 
    Details::Reader::_narrow(reader);
  if (dr == 0) 
  {
    std::cerr << "EchoWriter<DDS_STRUCT_T>::on_data_available: _narrow failed." 
              << std::endl;
    return;
  }

  // Take a sample

  DDS_STRUCT_T sample;
  DDS::SampleInfo si ;
  DDS::ReturnCode_t status = dr->take_next_sample(sample, si);

  while (status == DDS::RETCODE_OK) 
  {
    if (si.instance_state == DDS::ALIVE_INSTANCE_STATE)
    {
      if (this->timestampField_ == 0)
      {
        // Not saving a timestamp; just write it back right away
        this->writer_->write(sample,this->instance_handle_);
      }
      else if (sample.*(this->timestampField_) > 0)
      {
        MIDDLEWARENEWSBRIEF_PROFILER_TIME_TYPE now =
          MIDDLEWARENEWSBRIEF_PROFILER_GET_TIME;
  
        // Profile
        {
          MIDDLEWARENEWSBRIEF_PROFILE_POINT_W_START(this->profileString_.c_str(),sample.*(this->timestampField_)); 
        }

        // Save it off to echo it back later.  We're not trying to measure
        // backpressure
        this->waiting_for_echo_.push_back(
          SampleWrapper(sample,now,this->timestampField_));
      }
    }

    // See if there's another one
    status = dr->take_next_sample(sample, si);
  } 
}

} // MiddlewareNewsBrief

#endif 

