// Copyright (c) 2010, Object Computing, Inc.
// All rights reserved.

#ifdef _MSC_VER
# pragma once
#endif
#ifndef MIDDLEWARENEWSBRIEF_DDSUTIL_PROFILER_H
#define MIDDLEWARENEWSBRIEF_DDSUTIL_PROFILER_H

#include "MiddlewareNewsBrief_DdsUtil_Export.h"
#include <iostream>
#include <assert.h>

/// enable or disable generation and usage of profiler code.
#define MIDDLEWARENEWSBRIEF_PROFILER_ENABLE

// Note: boost's microsecond_clock was slow enough to skew the results
// Note: boost's microsecond_clock was slow enough to skew the results
// hence the win32 GetTickCount stuff.
#ifdef _WIN32
#include <windows.h>
# ifndef _WINBASE_ // if we didn't #include <windows.h>
// this is the ultimate expansion of the declaration for GetTickCount
// after the macros have been resolved on a 16 bit windows build
// With luck, it won't conflict with the windows declaration in case
// the included occur in the wrong order
typedef unsigned long DWORD;
__declspec(dllimport) DWORD GetTickCount(void);
# endif // _WINBASE_
#endif // _WIN32

#ifdef _WIN32
# define MIDDLEWARENEWSBRIEF_PROFILER_TIME_TYPE DWORD
# define MIDDLEWARENEWSBRIEF_PROFILER_TIME_UNITS "milliseconds"
#else
# include <boost/cstdint.hpp>
# include <sys/time.h>
# define MIDDLEWARENEWSBRIEF_PROFILER_TIME_TYPE boost::uint64_t
# define MIDDLEWARENEWSBRIEF_PROFILER_TIME_UNITS "microseconds"
#endif
#define MIDDLEWARENEWSBRIEF_PROFILER_GET_TIME MiddlewareNewsBrief::profilerGetTime()
#define MIDDLEWARENEWSBRIEF_PROFILER_DIFF(a, b) (a - b)
#define USEC_PER_SEC 1000000

namespace MiddlewareNewsBrief {

  inline
  MIDDLEWARENEWSBRIEF_PROFILER_TIME_TYPE profilerGetTime()
  {
    // Note: boost's microsecond_clock was slow enough to skew the results
    // hence the win32 GetTickCount stuff.
    // Millisecond resolution on Windows
#ifdef _WIN32
    return ::GetTickCount();
#else
    timeval tv;
    ::gettimeofday(&tv,0);

    return tv.tv_usec + (tv.tv_sec * USEC_PER_SEC);
#endif
  }

  class ProfileInstance;

  /// @brief Accumulate profiler statistics.
  ///
  /// A ProfileAccumulater is statically created for each Profile Point.
  /// The ProfileInstances created to do the actual timing store their results
  /// into the corresponding ProfileAccumulator.
  /// These accumulators link themselves together in a list starting at root_.
  /// Walking this list lets you find all profile points in the system.
  /// The static write(...) method writes a tab-delimited file of the statistics.
  /// Hint: try importing this file into a spreadsheet for analysis.
  class MiddlewareNewsBrief_DdsUtil_Export ProfileAccumulator
  {
  public:
    /// @brief Create the ProfileAccumulator
    /// @param name identifies the profile point.
    /// @param file should be generated by the __FILE__ predefined macro.
    /// @param line should be generated by the __LINE__ predefined macro.
    ProfileAccumulator(const char * name, const char * file, size_t line);
    /// @brief write in machine-readable form (tab delimited columns)
    static void write(std::ostream & out);
    /// @brief write in somewhat human readable format
    static void print(std::ostream & out);

    const char* name() const { return name_; }
    
  private:
    friend class MiddlewareNewsBrief::ProfileInstance;
    static ProfileAccumulator * root_;
    ProfileAccumulator * next_;
    const char * name_;
    const char * file_;
    size_t line_;
    size_t entries_;
    size_t exits_;
    size_t pauses_;
    size_t resumes_;
    MIDDLEWARENEWSBRIEF_PROFILER_TIME_TYPE sum_;
    MIDDLEWARENEWSBRIEF_PROFILER_TIME_TYPE sumOfSquares_;
    size_t recursions_;
    MIDDLEWARENEWSBRIEF_PROFILER_TIME_TYPE recursiveSum_;
    MIDDLEWARENEWSBRIEF_PROFILER_TIME_TYPE recursiveSumOfSquares_;

    MIDDLEWARENEWSBRIEF_PROFILER_TIME_TYPE total_jitter_;
    MIDDLEWARENEWSBRIEF_PROFILER_TIME_TYPE prev_lapse_for_jitter_;
    MIDDLEWARENEWSBRIEF_PROFILER_TIME_TYPE max_;

    static MIDDLEWARENEWSBRIEF_PROFILER_TIME_TYPE firstStart_;
    static MIDDLEWARENEWSBRIEF_PROFILER_TIME_TYPE lastEnd_;
  };

  /// @brief an auto variable to measure the time in a section of code
  ///
  /// Measures time from creation to destruction (usually controlled by scope)
  /// and stores the results in a ProfileAccumulator
  class MiddlewareNewsBrief_DdsUtil_Export ProfileInstance
  {
  public:

    /// @brief Construct and link to an accumulator
    /// @param accumulator to receive the measured results.
    ProfileInstance(ProfileAccumulator & accumulator)
      : accumulator_(accumulator)
      , start_(MIDDLEWARENEWSBRIEF_PROFILER_GET_TIME)
      , running_(true)
    {
      accumulator_.entries_ += 1;
    }

    /// @brief Construct and link to an accumulator
    /// @param accumulator to receive the measured results.
    ProfileInstance(ProfileAccumulator & accumulator, MIDDLEWARENEWSBRIEF_PROFILER_TIME_TYPE start)  // DCB
      : accumulator_(accumulator)
      , start_(start)
      , running_(true)
    {
      accumulator_.entries_ += 1;
    }

    /// @brief Stop timing and accumulate results.
    ~ProfileInstance()
    {
      stop();
      accumulator_.exits_ += 1;
    }

    /// @brief Stop timing -- may be resumable
    /// @returns false if it was already stopped.
    bool pause()
    {
      bool result = running_;
      stop();
      accumulator_.pauses_ += 1;
      return result;
    }

    /// @brief Resume timing after a pause
    /// @param pauseState is the return value from a pause
    void resume(bool pauseState)
    {
      accumulator_.resumes_ += 1;
      if(!running_ && pauseState)
      {
        start_ = MIDDLEWARENEWSBRIEF_PROFILER_GET_TIME;
        running_ = true;
      }
    }

  private:
    void stop()
    {
      if(running_)
      {
        MIDDLEWARENEWSBRIEF_PROFILER_TIME_TYPE now = MIDDLEWARENEWSBRIEF_PROFILER_GET_TIME;
        MIDDLEWARENEWSBRIEF_PROFILER_TIME_TYPE lapse = MIDDLEWARENEWSBRIEF_PROFILER_DIFF(now, start_);
        
        if (ProfileAccumulator::firstStart_ == 0
            || start_ < ProfileAccumulator::firstStart_)
        {
          ProfileAccumulator::firstStart_ = start_;
        }
        ProfileAccumulator::lastEnd_ = now;

        //if (lapse > 10000) {
          //accumulator_.entries_ -= 1;
          //accumulator_.exits_ -= 1;
          //return;
          //printf("%s: %u,%u = %d\n",accumulator_.name(),start_,now,lapse);
        //}
        
        accumulator_.sum_ += lapse;
        accumulator_.sumOfSquares_ += lapse * lapse;
        assert(accumulator_.entries_ > accumulator_.exits_);
        if(accumulator_.entries_ != accumulator_.exits_ + 1)
        {
          accumulator_.recursions_ += 1;
          accumulator_.recursiveSum_ += lapse;
          accumulator_.recursiveSumOfSquares_ += lapse * lapse;
        }
        else
        {
          // Maximum time lapse
          if (lapse > accumulator_.max_)
          {
            accumulator_.max_ = lapse;
          }
          // Calculate jitter
          if (accumulator_.prev_lapse_for_jitter_ > 0) {
            // Can't use abs() here on Windows
            if (lapse > accumulator_.prev_lapse_for_jitter_) {
              accumulator_.total_jitter_ +=
                lapse - accumulator_.prev_lapse_for_jitter_;
            } else {
              accumulator_.total_jitter_ +=
                accumulator_.prev_lapse_for_jitter_ - lapse;
            }
          }
          accumulator_.prev_lapse_for_jitter_ = lapse;
        }
        running_ = false;
      }
    }

  private:
    ProfileAccumulator & accumulator_;
    MIDDLEWARENEWSBRIEF_PROFILER_TIME_TYPE start_;
    bool running_;
  };
}

#ifdef MIDDLEWARENEWSBRIEF_PROFILER_ENABLE
/// Define the start point of a block of code to be profiled.
# define MIDDLEWARENEWSBRIEF_PROFILE_POINT(name)   \
    static MiddlewareNewsBrief::ProfileAccumulator PROFILE_accumulator(name, __FILE__, __LINE__); \
    MiddlewareNewsBrief::ProfileInstance PROFILE_instance(PROFILE_accumulator)

/// Define the start point of a block of code to be profiled.
// DCB
# define MIDDLEWARENEWSBRIEF_PROFILE_POINT_W_START(name,start)   \
    static MiddlewareNewsBrief::ProfileAccumulator PROFILE_accumulator(name, __FILE__, __LINE__); \
    MiddlewareNewsBrief::ProfileInstance PROFILE_instance(PROFILE_accumulator,start)

/// Pause or stop profiling
# define MIDDLEWARENEWSBRIEF_PROFILE_PAUSE \
    bool PROFILE_pauseState = PROFILE_instance.pause()

/// Resume after pause
# define MIDDLEWARENEWSBRIEF_PROFILE_RESUME \
    PROFILE_instance.pause(PROFILE_pauseState)

/// Define the start point of a block of code to be profiled.
/// Allows more than one profiler in the same scope.
# define MIDDLEWARENEWSBRIEF_NESTED_PROFILE_POINT(id, name) \
    static MiddlewareNewsBrief::ProfileAccumulator PROFILE_accumulator##id(name, __FILE__, __LINE__); \
    MiddlewareNewsBrief::ProfileInstance PROFILE_instance##id(PROFILE_accumulator##id)

/// Pause or stop profiling
# define MIDDLEWARENEWSBRIEF_NESTED_PROFILE_PAUSE(id) \
  bool PROFILE_pauseState##id = PROFILE_instance##id.pause()

/// Resume after pause
# define MIDDLEWARENEWSBRIEF_NESTED_PROFILE_RESUME(id) \
  PROFILE_instance##id.pause(PROFILE_pauseState##id)

#else // MIDDLEWARENEWSBRIEF_PROFILER_ENABLE

# define MIDDLEWARENEWSBRIEF_PROFILE_POINT(name)  void(0)
# define MIDDLEWARENEWSBRIEF_PROFILE_POINT_W_START(name,start) void(0)
# define MIDDLEWARENEWSBRIEF_PROFILE_PAUSE  void(0)
# define MIDDLEWARENEWSBRIEF_NESTED_PROFILE_POINT(id, name)  void(0)
# define MIDDLEWARENEWSBRIEF_NESTED_PROFILE_PAUSE(id) void(0)
# define MIDDLEWARENEWSBRIEF_NESTED_PROFILE_RESUME(id) void(0)
#endif // MIDDLEWARENEWSBRIEF_PROFILER_ENABLE

#endif // MIDDLEWARENEWSBRIEF_DDSUTIL_PROFILER_H
