// Copyright (c) 2010, Object Computing, Inc.
// All rights reserved.

#ifdef _MSC_VER
# pragma warning(disable:4005) // Disable VC warning about duplicate macros;
                               // Boost and ACE define same macros for Windows
#endif

#include "RawTypeDetails.h"
#include <RawTypeSupportImpl.h>

#include "DdsUtilPch.h"
#include "DDSUtil.h"
#include "DataWriter_T.h"
#include "DoneTokenManager.h"

// OpenDDS
#include <dds/DCPS/Service_Participant.h>
#include <dds/DCPS/Marked_Default_Qos.h>
#include <dds/DCPS/PublisherImpl.h>
#include <dds/DCPS/transport/framework/TheTransportFactory.h>

#include <iostream>
#include <set>

// Profiler
#include "Profiler.h"

void usage() {
  std::cout 
    << "-num numMessages   : Number of messages to write\n"
    << "-s messageSize     : Size of each message in bytes\n"
    << "-ns numNubscribers : number of subscribers\n"
    << std::endl;
}

int main(int argc, char** argv)
{
  const int DOMAIN_ID = 8675309;

  size_t numMessages = 10000;
  size_t messageSize = 1000;
  size_t numSubscribers = 1;

  // Command-Line Arguments
  //
  //    -num numMessages : Number of messages to write
  //    -s   messageSize : Size of each message in bytes
  //    -ns numNubscribers : number of subscribers
  //
  int i = 0;
  while (i < argc) {
    if (std::string(argv[i]) == "-num" && (i+1) < argc) {
      numMessages = boost::lexical_cast<size_t>(argv[++i]);
    } else if (std::string(argv[i]) == "-s" && (i+1) < argc) {
       messageSize = boost::lexical_cast<size_t>(argv[++i]);
    } else if (std::string(argv[i]) == "-ns" && (i+1) < argc) {
       numSubscribers = boost::lexical_cast<size_t>(argv[++i]);
    } else if (std::string(argv[i]) == "-?" ) {
      usage();
      return 0;
    }
    ++i;
  }

  std::cout << "Number of messages is " << numMessages << std::endl;
  std::cout << "Messages size is " << messageSize << std::endl;

  try
  {
    //
    // OpenDDS Init
    //

    // Initialize, and create a DomainParticipantFactory
    DDS::DomainParticipantFactory_var factory = 
      TheParticipantFactoryWithArgs(argc, argv);

    //  Create the DomainParticipant
    DDS::DomainParticipant_var participant = 
      factory->create_participant(DOMAIN_ID,
                                  PARTICIPANT_QOS_DEFAULT,
                                  0,
                                  0);
    if (participant == 0) 
    {
      std::cerr << "create_participant failed." << std::endl;
      return -1;
    }

    // Create a publisher for the topics
    const int TRANSPORT_IMPL_ID = 1;
    DDS::Publisher_var publisher = 
      MiddlewareNewsBrief::DDSUtil::create_publisher(participant.in(),
                                                     TRANSPORT_IMPL_ID,
                                                     "Primary");

    // Create a subscriber for the Echo of the two topics
    const int TRANSPORT_IMPL_ID_2 = 2;
    DDS::Subscriber_var subscriber = 
      MiddlewareNewsBrief::DDSUtil::create_subscriber(participant.in(),
                                                      TRANSPORT_IMPL_ID_2,
                                                      "Secondary");

    // Initialize the DoneToken manager, which publishes a "done" token
    MiddlewareNewsBrief::DoneTokenManager doneToken;
    doneToken.initWriter(participant,publisher,subscriber);

    //
    // DataWriters, one per message type
    //

    typedef MiddlewareNewsBrief::RawBuffer RawBuffer;
    boost::shared_ptr<MiddlewareNewsBrief::DataWriter<RawBuffer> > 
      rawDataWriter(new MiddlewareNewsBrief::DataWriter<RawBuffer>());
    rawDataWriter->init(participant, 
                        publisher, 
                        subscriber);
    rawDataWriter->sleepUsecBetweenWrites(0);

    boost::this_thread::sleep(boost::posix_time::seconds(3));

    MiddlewareNewsBrief::RawBuffer payload;
    payload.buffer.length(messageSize);    

    std::cout << "Writing..." << std::endl;

    MIDDLEWARENEWSBRIEF_PROFILER_TIME_TYPE start = 
      MIDDLEWARENEWSBRIEF_PROFILER_GET_TIME;

    for (size_t i = 0; i < numMessages; ++i) 
    {
      rawDataWriter->onMessage(payload
#ifdef MIDDLEWARENEWSBRIEF_PROFILER_ENABLE
                               ,0
#endif
                               );
      for (size_t jj = 0; jj < numSubscribers; ++jj) {      
        rawDataWriter->recv_echo();
      }
    }

    MIDDLEWARENEWSBRIEF_PROFILER_TIME_TYPE end = 
      MIDDLEWARENEWSBRIEF_PROFILER_GET_TIME;

    double latency = (double)(end-start) / ((double)numMessages * 2.0)  / (double)numSubscribers;
    std::cout << "\n\nAverage latency in " 
              << MIDDLEWARENEWSBRIEF_PROFILER_TIME_UNITS
              << ": " << latency << "\n\n" << std::endl;

    doneToken.writeDone("done");

    // Give the samples time to make it to the subscribers
    DDS::Duration_t duration;
    duration.sec = DDS::DURATION_INFINITE_SEC;
    duration.nanosec = DDS::DURATION_INFINITE_NSEC;
    publisher->wait_for_acknowledgments(duration);

    // OpenDDS Cleanup
    doneToken.fini(publisher.in());
    if (publisher != 0) 
    {
      publisher->delete_contained_entities();
      participant->delete_publisher(publisher.in());
      publisher = DDS::Publisher::_nil();
    }
    if (participant != 0) 
    {
      participant->delete_contained_entities();
    }
    if (factory != 0) 
    {
      factory->delete_participant(participant.in ());
    }
    TheTransportFactory->release();
    TheServiceParticipant->shutdown();  
  }
  catch (std::exception & e)
  {
    std::cerr << "Exception: " << e.what() << std::endl;
    return -1;
  }
  catch (CORBA::Exception& e) 
  {
    std::cerr << "Exception: " << e << std::endl;
    return -1;
  }
  return 0;
}

