﻿// Copyright (c) 2010, Object Computing, Inc.
// All rights reserved.

using System;
using System.Collections.Generic;
using System.Text;
using System.Runtime.InteropServices;
using System.Diagnostics;
using System.Runtime.Serialization;
using System.Runtime.Serialization.Formatters.Binary;
using System.IO;
using System.Threading;
using MiddlewareNewsBrief;

public class TypedDotnetPublisher
{
   OpenDdsPubSubProxy proxy_;
   uint echo_count_ = 0;
   uint message_count_ = 0;

   int marketDataWriterHandle_ = -1;
   int quoteRequestDataWriterHandle_ = -1;
   IFormatter formatter_ = new BinaryFormatter();

   static unsafe int Main(string[] args)
   {
      TypedDotnetPublisher me = new TypedDotnetPublisher();
      me.Run(args);
      return 0;
   }

   public TypedDotnetPublisher()
   {
   }

   int Run(string[] args)
   {
      const String QUOTE_TOPIC = "QuoteRequest";
      const String MARKET_DATA_TOPIC = "MarketData";

      int roundtripCount = 10000;
      int num_subscribers = 1;

      for (int i = 0; i < args.Length; ++i)
      {
         if (args[i].Equals("-num"))
         {
            roundtripCount = Convert.ToInt32(args[i + 1]);
         } 
         else if (args[i].Equals("-ns"))
         {
            num_subscribers = Convert.ToInt32(args[i + 1]);
         }
      }

      Console.WriteLine(num_subscribers + " Subscribers, " + roundtripCount + " messages");

      this.proxy_ = new OpenDdsPubSubProxy(MiddlewareNewsBrief.ProcessType.PUBLISHER_PROCESS);
      this.proxy_.ProcessNotification += new OpenDdsPubSubProxy.EventHandler(OnEchoReceived);

      this.marketDataWriterHandle_ = this.proxy_.createDataWriter(MARKET_DATA_TOPIC);
      this.quoteRequestDataWriterHandle_ = this.proxy_.createDataWriter(QUOTE_TOPIC);

      System.Threading.Thread.Sleep(1000);
      
      Console.Out.WriteLine("Sending messages -- " + num_subscribers + " subscribers, "
         + roundtripCount + " messages");

      // Create two messages to send
      MarketData md = MarketData.createTestData();
      QuoteRequest qr = QuoteRequest.createTestData();

      //  Start measuring the time.
      System.Diagnostics.Stopwatch watch;
      watch = new Stopwatch();
      watch.Start();

      //  Start sending messages.
      for (uint i = 0; i < roundtripCount; i++)
      {
         this.message_count_ = i;
         
         // Send 90% MarketData messages
         if (i % 10 == 5)
         {
            qr.isEcho = false;
            qr.counter = i;
            byte[] quoteMsg = serialize(qr, formatter_);

            proxy_.write(this.quoteRequestDataWriterHandle_, quoteMsg);
         }
         else
         {
            md.isEcho = false;
            md.counter = i;
            byte[] mdMsg = serialize(md, formatter_);

            proxy_.write(this.marketDataWriterHandle_, mdMsg);
         }

         // Wait for echo from each subscriber
         for (int jj = 0; jj < num_subscribers; ++jj)
         {
            this.TakeEcho();
         }
      }

      //  Stop measuring the time.
      watch.Stop();
      Int64 elapsedTime = watch.ElapsedTicks;

      this.proxy_.writeDone();

      //  Print out the test parameters.
      Console.Out.WriteLine("roundtrip count: " + roundtripCount);

      //  Compute and print out the latency.
      double latency = (double)(elapsedTime) / roundtripCount / 2 *
          1000000 / Stopwatch.Frequency / (double)num_subscribers;
      Console.Out.WriteLine("\n\nYour average latency is {0} [us]\n\n",
          latency.ToString("f2"));

      this.proxy_.fini();
      return 0;
   }

   // invoked when a DDS sample arrives
   void OnEchoReceived(object parent, int data_writer_handle, byte[] buffer)
   {
      // deserialize the buffer into a Quote or MarketData object

      MemoryStream stream = new MemoryStream(buffer);
      object obj = this.formatter_.Deserialize(stream);
      stream.Close();

      if (data_writer_handle == this.quoteRequestDataWriterHandle_)
      {
         QuoteRequest qr = (QuoteRequest)obj;
         Debug.Assert(qr.isEcho == true, "Subscriber forgot to set isEcho flag to true");
         Debug.Assert(qr.counter == this.message_count_, "Counter mismatch in subscriber's reply");
      }
      else if (data_writer_handle == this.marketDataWriterHandle_)
      {
         MarketData md = (MarketData)obj;
         Debug.Assert(md.isEcho == true, "Subscriber forgot to set isEcho flag to true");
         Debug.Assert(md.counter == this.message_count_, "Counter mismatch in subscriber's reply");
      }
      else
      {
         Console.WriteLine("OnEchoReceived: no data_writer_handle for " + data_writer_handle);
         return;
      }
    
      lock (this)
      {
         ++this.echo_count_;
         Monitor.PulseAll(this);
      }
   }

   void TakeEcho()
   {
      lock (this)
      {
         while (this.echo_count_ == 0) {
            Monitor.Wait(this);
         }
         --this.echo_count_;
      }
   }

    static byte[] serialize(object obj, IFormatter formatter)
    {
       MemoryStream stream = new MemoryStream();
       formatter.Serialize(stream, obj);
       stream.Close();
       return stream.ToArray();
    }

    static object deserialize(byte[] msg, String topic, IFormatter formatter)
    {
       MemoryStream stream = new MemoryStream(msg);

       // Seek past "topic" for ZeroMQ
       stream.Seek(topic.Length + 1, SeekOrigin.Begin);
       object obj = formatter.Deserialize(stream);
       stream.Close();
       return obj;
    }
}
