// Copyright (c) 2010, Object Computing, Inc.
// All rights reserved.

#ifdef _MSC_VER
# pragma warning(disable:4996) // Disable VC warning from Boost serialization
# pragma warning(disable:4099) // Disable VC warning from Boost serialization
#endif

#include "zmq.hpp"
#include "MarketData.pb.h"
#include <iostream>

pbMarketData createMarketData();
pbQuoteRequest createQuoteRequest();

const std::string MARKET_DATA_TOPIC = "MarketData";
const std::string QUOTE_REQUEST_TOPIC = "QuoteRequest";

int main (int argc, char *argv [])
{
  try {
    if (argc < 4) {
      std::cout << "usage: ZeromqTypedProtobufPublisher <roundtrip-count>  <bind-to-sub> [<connect-to-pub>]+\n"
                << "e.g.:  ZeromqTypedProtobufPublisher  10000 tcp://eth0:54321 tcp://spider:54322 tcp://spider:54323\n"
                << "          use a literal IP address and port for each endpoint\n"
                << std::endl;
    }
    size_t roundtrip_count = atoi (argv [1]);
    const char* bind_to_sub = argv[2];

    zmq::context_t ctx (1, 1);
    zmq::socket_t pub(ctx,ZMQ_PUB);
    zmq::socket_t sub(ctx,ZMQ_SUB);
    sub.setsockopt(ZMQ_SUBSCRIBE,"MarketData\x00",11);
    sub.setsockopt(ZMQ_SUBSCRIBE,"QuoteRequest\x00",13);
    sub.bind(bind_to_sub);

    size_t num_subscribers = 0;
    for (int argi = 3; argi < argc; ++argi) {
      pub.connect(argv[argi]);
      ++num_subscribers;
    }

    printf("Entering send loop -- %d messages\n", roundtrip_count);

    pbMarketData md = createMarketData();
    pbQuoteRequest qr = createQuoteRequest();

    void* watch = zmq_stopwatch_start ();

    for (size_t i = 0; i < roundtrip_count; i++) 
    {
      // Send 10 MarketDatas for each QuoteRequest
      const std::string* topic_name;
      google::protobuf::Message* payload;
      if (i % 10 == 5) 
      {
        topic_name = &QUOTE_REQUEST_TOPIC;
        qr.set_is_echo(false);
        qr.set_counter(i);
        payload = &qr;
      }
      else 
      {
        topic_name = &MARKET_DATA_TOPIC;
        md.set_is_echo(false);
        md.set_counter(i);
        payload = &md;
      }

      size_t request_length = payload->ByteSize() + topic_name->length() + 1;
      zmq::message_t msg(request_length);
      memset(msg.data(),0,msg.size());
      memcpy(msg.data(),topic_name->c_str(),topic_name->length());
      payload->SerializeToArray(reinterpret_cast<char*>(msg.data())+topic_name->length()+1,payload->ByteSize());

      pub.send(msg);

      for (size_t jj = 0; jj < num_subscribers; ++jj) 
      {
        // Wait for echoed message from each subscriber
        zmq::message_t reply_msg;
        sub.recv(&reply_msg);

        char* reply_buffer = reinterpret_cast<char*>(reply_msg.data());
        
        // "topic" name should be first N characters, null-terminated
        std::string reply_topic_name(reply_buffer);

        if (reply_topic_name == MARKET_DATA_TOPIC) 
        {
          pbMarketData mdReply;
          mdReply.ParseFromArray(reply_buffer + reply_topic_name.length() + 1,
                                 reply_msg.size() - (reply_topic_name.length() + 1));

          if (mdReply.is_echo() == false || mdReply.counter() != i)
          {
            std::cerr << "MarketData reply isEcho or counter mismatch" << std::endl;
          }
        } 
        else if (reply_topic_name == QUOTE_REQUEST_TOPIC)
        {
          pbQuoteRequest qrReply;
          qrReply.ParseFromArray(reply_buffer + reply_topic_name.length() + 1,
                                 reply_msg.size() - (reply_topic_name.length() + 1));

          if (qrReply.is_echo() == false || qrReply.counter() != i)
          {
            std::cerr << "QuoteRequest reply isEcho or counter mismatch" << std::endl;
          }
        }
        else 
        {
          std::cerr << "Received invalid topic name: " << reply_topic_name.c_str() << std::endl;
          return -1;
        }
      }
    }

    unsigned long elapsed = zmq_stopwatch_stop (watch);
    double latency = (double) elapsed / (roundtrip_count * 2.0) / (double)num_subscribers;

    printf ("roundtrip count: %d\n", (int) roundtrip_count);
    printf ("\n\naverage latency: %.3f [us]\n\n\n", (double) latency);
  
    return 0;
  } catch (std::exception &e) {
    std::cout << "An error occurred: " << e.what() << std::endl;
    return 1;
  }
}


void populateMarketDataEntry(pbMarketDataEntry& entry)
{
  entry.set_mdupdateaction(1);
  entry.set_mdpricelevel(2);
  entry.set_mdentrytype("7");
  entry.set_openclosesettleflag(3);
  entry.set_securityid(99);
  entry.set_securityidsource(9);
  entry.set_rptseq(2);
  entry.set_mdentrypx(100.0);
  entry.set_mdentrytime(12345);
  entry.set_mdentrysize(50);
  entry.set_numberoforders(10);
  entry.set_tradingsessionid("2");
  entry.set_netchgprevday(10.0);
  entry.set_tradevolume(30);
  entry.set_tradecondition("W");
  entry.set_tickdirection("0");
  entry.set_quotecondition("C");
  entry.set_aggressorside(2);
  entry.set_matcheventindicator("1");
}


pbMarketData createMarketData()
{
  pbMarketData data;

  data.set_is_echo(false);
  data.set_counter(0);
  
  data.set_securityid(2112);
  data.set_applversionid("1.0");
  data.set_messagetype("100");
  data.set_sendercompid("Test Exchange");
  data.set_msgseqnum(4);
  data.set_sendingtime(00162635);
  data.set_tradedate(20100422);
  
  populateMarketDataEntry(*(data.add_mdentries()));
  populateMarketDataEntry(*(data.add_mdentries()));
  populateMarketDataEntry(*(data.add_mdentries()));
  
  return data;
}  

void populateRelatedSym(pbRelatedSym& sym)
{
  sym.set_symbol("[N/A]");
  sym.set_orderquantity(25);
  sym.set_side(1);
  sym.set_transacttime(00162635);
  sym.set_quotetype(1);
  sym.set_securityid(99);
  sym.set_securityidsource(9);
}

pbQuoteRequest createQuoteRequest()
{
  pbQuoteRequest req;

  req.set_is_echo(false);
  req.set_counter(0);

  req.set_securityid(2112);
  req.set_applversionid("1.0");
  req.set_messagetype("100");
  req.set_sendercompid("Test Exchange");
  req.set_msgseqnum(4);
  req.set_sendingtime(00162635);
  req.set_quotereqid("R");
    
  populateRelatedSym(*(req.add_related()));
  populateRelatedSym(*(req.add_related()));
  populateRelatedSym(*(req.add_related()));
  
  return req;
}
