﻿// Copyright (c) 2010, Object Computing, Inc.
// All rights reserved.

using System;
using System.Collections.Generic;
using System.Text;
using System.Runtime.InteropServices;
using System.Diagnostics;
using System.Runtime.Serialization;
using System.Runtime.Serialization.Formatters.Binary;
using System.IO;
using MiddlewareNewsBrief;

class ZmqTypedSubscriber
{
    static unsafe int Main(string[] args)
    {
        const String QUOTE_TOPIC = "QuoteRequest";
        const String MARKET_DATA_TOPIC = "MarketData";

        if (args.Length != 3)
        {
           Console.Out.WriteLine("usage: ZmqTypedSubscriber <roundtrip-count> " +
                "<bind-to-sub> <connect-to-pub>\n");
           Console.Out.WriteLine(" e.g.: ZmqTypedSubscriber 10000 tcp://10.201.200.72:54322 tcp://10.201.200.72:54321\n");
           return 1;
        }

        int roundtripCount = Convert.ToInt32(args[0]);
        String bind_to_sub = args[1];
        String connect_to_pub = args[2];

        //  Initialise 0MQ infrastructure
        ZMQ.Context ctx = new ZMQ.Context(1, 1, 0);
        
        ZMQ.Socket pub = ctx.Socket(ZMQ.PUB);
        pub.Connect(connect_to_pub);

        ZMQ.Socket sub = ctx.Socket(ZMQ.SUB);
        sub.SetSockOpt(ZMQ.SUBSCRIBE, QUOTE_TOPIC);
        sub.SetSockOpt(ZMQ.SUBSCRIBE, MARKET_DATA_TOPIC);
        sub.Bind(bind_to_sub);

        ASCIIEncoding encoding = new System.Text.ASCIIEncoding();
        IFormatter formatter = new BinaryFormatter();

        Console.Out.WriteLine("Entering recv loop -- " 
           + roundtripCount + " messages");
        for (int i = 0; i < roundtripCount; i++)
        {
            byte[] msg;
            sub.Recv(out msg);

            // Get the "Topic" from the front of the byte array
            int topicEndIndex = Array.IndexOf(msg, (byte)'\x00');
            String topic = new String(encoding.GetChars(msg, 0, topicEndIndex));

            MemoryStream inStream = new MemoryStream(msg);
            inStream.Seek(topic.Length + 1, SeekOrigin.Begin);

            object obj = formatter.Deserialize(inStream);
            inStream.Close();

            if (topic.Equals(MARKET_DATA_TOPIC))
            {
               MarketData md = (MarketData)obj;
               if (md.isEcho == true) { Console.Out.WriteLine("Subscriber received echo sample"); }
               if (md.counter != i) { Console.Out.WriteLine("Counter mismatch"); }
               md.isEcho = true;
            }
            else if (topic.Equals(QUOTE_TOPIC))
            {
               QuoteRequest qr = (QuoteRequest)obj;
               if (qr.isEcho == true) { Console.Out.WriteLine("Subscriber received echo sample"); }
               if (qr.counter != i) { Console.Out.WriteLine("Counter mismatch"); }
               qr.isEcho = true;
            }
            else
            {
               Console.Out.WriteLine("ERROR: received topic " + topic);
               return -1;
            }

            MemoryStream outStream = new MemoryStream();

            // "topic" for ZeroMQ
            outStream.Write(new System.Text.ASCIIEncoding().GetBytes(topic + '\x00'),
                            0,
                            topic.Length + 1);

            formatter.Serialize(outStream, obj);
            outStream.Close();
            msg = outStream.ToArray();

            // Echo it back
            pub.Send(msg);   
        }
        Console.Out.WriteLine("Finished receiving messages");

        return 0;
    }
}
