// Copyright (c) 2010, Object Computing, Inc.
// All rights reserved.

#ifdef _MSC_VER
# pragma once
#endif
#ifndef FASTTODDS_EXAMPLES_DATAWRITER_T_H
#define FASTTODDS_EXAMPLES_DATAWRITER_T_H

#include "DDSUtil.h"
#include "EchoReceiver_T.h"
#include "DDSTypeDetails_T.h"
#include "Listener_T.h"
#include <string>

// OpenDDS
#include <dds/DCPS/Service_Participant.h>
#include <dds/DCPS/PublisherImpl.h>

// Profiler
#include "Profiler.h"

namespace MiddlewareNewsBrief{

    template <typename DDS_STRUCT_T>
    class DataWriter : public MiddlewareNewsBrief::Listener<DDS_STRUCT_T>
    {
    public:
      typedef MiddlewareNewsBrief::DDSTypeDetails<DDS_STRUCT_T> Details;

      DataWriter();
      virtual ~DataWriter();

      void init(DDS::DomainParticipant_ptr participant,
                DDS::Publisher_ptr publisher,
                DDS::Subscriber_ptr subscriber,
                CORBA::ULong DDS_STRUCT_T::* timestampField = 0,
                DDS_STRUCT_T instancePrototype = DDS_STRUCT_T(),
                std::string topic_name = "");

      void sleepUsecBetweenWrites(size_t usec) {
        this->sleepUsecBetweenWrites_ = usec;
      }

      void recv_echo();

      //
      // Implement Listener<DDS_STRUCT_T>
      //

      virtual void onMessage(DDS_STRUCT_T& message
#ifdef MIDDLEWARENEWSBRIEF_PROFILER_ENABLE
                             ,MIDDLEWARENEWSBRIEF_PROFILER_TIME_TYPE startTime
#endif
                             );

    private:
      DDS::InstanceHandle_t instance_handle_;
      
      // OpenDDS
      typename DDSTypeDetails<DDS_STRUCT_T>::Writer::_var_type writer_;
      MiddlewareNewsBrief::EchoReceiver<DDS_STRUCT_T> echo_receiver_;

      CORBA::ULong DDS_STRUCT_T::* timestampField_;
      std::string profileString_;
      
      size_t sleepUsecBetweenWrites_;
    };



template <typename DDS_STRUCT_T>
DataWriter<DDS_STRUCT_T>::DataWriter()
  : timestampField_(0)
  , sleepUsecBetweenWrites_(0)
{
}

template <typename DDS_STRUCT_T>
DataWriter<DDS_STRUCT_T>::~DataWriter()
{
}

template <typename DDS_STRUCT_T>
void
DataWriter<DDS_STRUCT_T>::init(
  DDS::DomainParticipant_ptr participant,
  DDS::Publisher_ptr publisher,
  DDS::Subscriber_ptr subscriber,
  CORBA::ULong DDS_STRUCT_T::* timestampField,
  DDS_STRUCT_T instancePrototype,
  std::string topic_name)
{
  this->timestampField_ = timestampField;
  this->profileString_ = "<" + Details::type_name + ">::endMessage()";
  
  // Create a DataWriter for the topic, and make sure it is non-nil
  DDS::DataWriter_var dw =
    DDSUtil::create_datawriter<typename Details::TypeSupportImpl>(participant,
                                                                  publisher,
                                                                  topic_name);

  // narrow the data writer created above 
  // and make sure the narrow's result is non-nil.
  this->writer_ = Details::Writer::_narrow(dw.in());
  if (this->writer_ == 0) 
  {
    std::ostringstream msg;
    msg  << topic_name.c_str() << " DataWriter could not be narrowed"
         << std::ends;
    throw std::runtime_error(msg.str());
  }

  this->instance_handle_ = this->writer_->register_instance(instancePrototype);

  // Write a sample to ensure that the transport is fully initialized
  DDS_STRUCT_T test_sample = instancePrototype;
  if (this->timestampField_ != 0) {
    test_sample.*(this->timestampField_) = 0;
    this->writer_->write(test_sample,this->instance_handle_);
  }

  this->echo_receiver_.init(participant,
                            subscriber,
                            this->writer_.in(),
                            this->timestampField_);
}


template <typename DDS_STRUCT_T>
void
DataWriter<DDS_STRUCT_T>::onMessage(DDS_STRUCT_T& message
#ifdef MIDDLEWARENEWSBRIEF_PROFILER_ENABLE
                                   ,MIDDLEWARENEWSBRIEF_PROFILER_TIME_TYPE startTime
#endif
                                   )
{
  { MIDDLEWARENEWSBRIEF_PROFILE_POINT_W_START(this->profileString_.c_str(), startTime); }

  if (this->sleepUsecBetweenWrites_ > 0)
  { 
    boost::this_thread::sleep
      (boost::posix_time::microseconds(this->sleepUsecBetweenWrites_));
  }

#ifdef MIDDLEWARENEWSBRIEF_PROFILER_ENABLE
  if (this->timestampField_ != 0) {
    message.*(this->timestampField_) = MIDDLEWARENEWSBRIEF_PROFILER_GET_TIME;
  }
#endif

  this->writer_->write(message,this->instance_handle_);
}


template <typename DDS_STRUCT_T>
void
DataWriter<DDS_STRUCT_T>::recv_echo()
{
  this->echo_receiver_.recv_echo();
}


} // MiddlewareNewsBrief

#endif 

