#include <exception>
#include <dds/DdsDcpsInfrastructureC.h>
#include <dds/DdsDcpsPublicationC.h>
#include <dds/DCPS/Marked_Default_Qos.h>
#include <dds/DCPS/Service_Participant.h>
#include <dds/DCPS/WaitSet.h>
#include <dds/DCPS/transport/framework/TheTransportFactory.h>
#include "MessengerTypeSupportImpl.h"

// based on dev guide Messenger example

class MyException : public std::exception {
	std::string _what;
public:
	MyException(const char *what) : _what(what) {}
	~MyException() throw() {}
	const char *what() { return _what.c_str(); }
};

void Sleep(int ms) {
	ACE_Time_Value tv;
	tv.set(ms / 1000.0);
	ACE_OS::sleep(tv); 
}


int ACE_TMAIN(int argc, ACE_TCHAR *argv[]) {

	try {
		DDS::DomainParticipantFactory_var dpf =
			TheParticipantFactoryWithArgs(argc, argv);

		// create domain participant
		DDS::DomainParticipant_var participant =
			dpf->create_participant(42,
			PARTICIPANT_QOS_DEFAULT,
			0,
			OpenDDS::DCPS::DEFAULT_STATUS_MASK);

		if (0 == participant) 
			throw MyException("create_participant failed");


		// register type
		Messenger::MessageTypeSupport_var ts =
			new Messenger::MessageTypeSupportImpl();

		if (ts->register_type(participant.in(), "") != DDS::RETCODE_OK) 
			throw MyException("reigster_type failed");


		// create topic
		CORBA::String_var type_name = ts->get_type_name();
		DDS::Topic_var topic =
			participant->create_topic("MyTopic",
			type_name.in(),
			TOPIC_QOS_DEFAULT,
			0,
			OpenDDS::DCPS::DEFAULT_STATUS_MASK);

		if (0 == topic) 
			throw MyException("create_topic failed");


		// create publisher
		DDS::Publisher_var publisher =
			participant->create_publisher(PUBLISHER_QOS_DEFAULT,
			0,
			OpenDDS::DCPS::DEFAULT_STATUS_MASK);

		if (0 == publisher) 
			throw MyException("create_publisher failed");


		// create and attach the transport
		OpenDDS::DCPS::TransportImpl_rch transport_impl =
			TheTransportFactory->create_transport_impl(OpenDDS::DCPS::DEFAULT_SIMPLE_TCP_ID,
			OpenDDS::DCPS::AUTO_CONFIG);

		if (transport_impl->attach(publisher.in()) != OpenDDS::DCPS::ATTACH_OK) 
			throw MyException("transport creation failed");


		// create and narrow datawriter
		DDS::DataWriter_var writer =
			publisher->create_datawriter(topic.in(),
			DATAWRITER_QOS_DEFAULT,
			DDS::DataWriterListener::_nil(),
			OpenDDS::DCPS::DEFAULT_STATUS_MASK);

		if (0 == writer) 
			throw MyException("create_datawriter failed");

		Messenger::MessageDataWriter_var message_writer =
			Messenger::MessageDataWriter::_narrow(writer.in());

		if (0 == message_writer) 
			throw MyException("writer _narrow failed");


		// block until a subscriber is available
		DDS::StatusCondition_var condition = writer->get_statuscondition();
		condition->set_enabled_statuses(DDS::PUBLICATION_MATCHED_STATUS);

		DDS::WaitSet_var ws = new DDS::WaitSet;
		ws->attach_condition(condition);

		DDS::ConditionSeq conditions;
		DDS::PublicationMatchedStatus matches = { 0, 0, 0, 0, 0 };
		DDS::Duration_t timeout = { 30, 0 };

		do {
			if (ws->wait(conditions, timeout) != DDS::RETCODE_OK) 
				throw MyException("wait failed");

			if (writer->get_publication_matched_status(matches) != ::DDS::RETCODE_OK)
				throw MyException("get_publication_matched_status failed"); 

		} while (matches.current_count < 1);

		ws->detach_condition(condition);

		// write samples
		Messenger::Message message;
		for (int i = 10; i >=0 ; i--) {
			message.id = i;
			message.msg = "Message";

			std::cout << "[Publisher] Publish: msg='" << message.msg.in() << "' id=" << message.id << std::endl;
			DDS::ReturnCode_t err = message_writer->write(message, DDS::HANDLE_NIL);
			if (err!=DDS::RETCODE_OK)
				throw MyException("write failed");

			Sleep(1000);
		}

		// wait for samples to be acknowledged
		if (message_writer->wait_for_acknowledgments(timeout) != DDS::RETCODE_OK) 
			throw MyException("wait_for_acknowledgments failed"); 

		// clean up
		participant->delete_contained_entities();
		dpf->delete_participant(participant.in());

		TheTransportFactory->release();
		TheServiceParticipant->shutdown();

	} catch (const CORBA::Exception& e) {
		e._tao_print_exception("Exception caught in main():");
		return -1;
	} catch (const std::exception& e) {
		std::cerr << "Exception caught in main(): " << e.what() << std::endl;
		return -1;
	}

	return 0;

}