#include "DDSMPI_ByTag.h"
#include "DDSException.h"
#include "GetArgs.h"
#include "dds/DCPS/Marked_Default_Qos.h"
#include "dds/DCPS/Service_Participant.h"
#include "dds/DCPS/transport/framework/TransportRegistry.h"


void DDSMPI_ByTag::Initialize(int &argc, ACE_TCHAR *argv[]) {
    BaseInitialize(argc, argv);

    // topic represents a communicator, so, for now, use one topic to represent MPI_COMM_WORLD
    MPIMessage::MessageByTagTypeSupport_var ts = new MPIMessage::MessageByTagTypeSupportImpl;
    if (ts->register_type(_dp, "") != DDS::RETCODE_OK)
        throw DDSException(_rank, "DDSMPI_ByTag::Initialize(): reigster_type() failed");

    CORBA::String_var type_name = ts->get_type_name();
    _topic = _dp->create_topic("MPI_COMM_WORLD", type_name, TOPIC_QOS_DEFAULT, 0, OpenDDS::DCPS::DEFAULT_STATUS_MASK);
    if (0 == _topic) 
        throw DDSException(_rank, "DDSMPI_ByTag::Initialize(): create_topic() failed");


    // create one publisher with the partition set as its rank
    DDS::StringSeq part;
    DDS::PublisherQos pub_qos;
    _dp->get_default_publisher_qos(pub_qos);
    part.length(1);
    std::ostringstream name;
    name << _rank;
    part[0] = name.str().c_str();
    pub_qos.partition.name = part;
    _pub = _dp->create_publisher(pub_qos, 0, OpenDDS::DCPS::DEFAULT_STATUS_MASK);
    if (0 == _pub) 
        throw DDSException(_rank, "DDSMPI_ByTag::Initialize(): create_publisher() failed");
    OpenDDS::DCPS::TransportRegistry::instance()->bind_config("c", _pub);

    _mdw = MPIMessage::MessageByTagDataWriter::_narrow(CreateDataWriter(_pub, _topic));
    if (0 == _mdw)
        throw DDSException(_rank, "DDSMPI_ByTag::Initialize(): writer _narrow() failed");


    // create one receiver for each sender (use partition) - when a specific source is desired
    // create another receiver that matches all senders (use partition) - for MPI_ANY_SOURCE

    DDS::SubscriberQos sub_qos;
    for (int i=0; i<=_size; i++) {
        _dp->get_default_subscriber_qos(sub_qos);
        part.length(1);
        std::ostringstream senderRank;
        if (i==_size)
            senderRank << "*";
        else
            senderRank << i;
        part[0] = senderRank.str().c_str();  // receive samples for the particular rank of the subscriber
        sub_qos.partition.name = part;
        DDS::Subscriber_var sub = _dp->create_subscriber(sub_qos, 0, OpenDDS::DCPS::DEFAULT_STATUS_MASK);
        if (0 == sub) 
		    throw DDSException(_rank, "DDSMPI_ByTag::Initialize(): create_subscriber() failed");
        OpenDDS::DCPS::TransportRegistry::instance()->bind_config("c", sub);
        _sub.push_back(sub);

        MPIMessage::MessageByTagDataReader_var mdr = MPIMessage::MessageByTagDataReader::_narrow(CreateDataReader(sub, _topic));
        if (0 == mdr) 
            throw DDSException(_rank, "DDSMPI_ByTag::Initialize(): reader _narrow() failed");
        _mdr.push_back(mdr);   // 0..size-1 is sender 0..size-1, size is any sender

        // add an instance list for each reader
        _readerInstances[i] = std::map<int, DDS::InstanceHandle_t>();
    }

    // WAIT FOR ASSOCIATIONS 
    WaitForPublicationCount(_mdw, 2*_size);  // matches both subscriber for the specific rank and "any sender" (so 2), for each process

    for (int i=0; i<=_size; i++)
        WaitForSubscriptionCount(_mdr[i], (i ==_size) ? _size : 1);
}




void DDSMPI_ByTag::Shutdown() {
    // wait until all messages have been acknowledged
    DDS::Duration_t fivesec = {5, 0};
    _mdw->wait_for_acknowledgments(fivesec);

    // delete data readers
    for (int i=0; i<=_size; i++) {
        if (_sub[i]->delete_datareader(_mdr[i]) != DDS::RETCODE_OK)
            throw DDSException(_rank, "DDSMPI_ByTag::Shutdown(): _sub[i]->delete_datareader() failed");
    }
    _sub.clear();

    // wait until the datawriter has no publications (all other subscribers have deleted their datareader)
    WaitForPublicationCount(_mdw, 0);

    // delete the remaining entities and shut down
    MPIMessage::MessageByTag sample;
    for (std::map<int, DDS::InstanceHandle_t>::iterator it = _writerInstances.begin(); it!=_writerInstances.end(); it++) {
        sample.tag = it->first;
        _mdw->unregister_instance(sample, it->second);
    }
    _writerInstances.clear();

    BaseShutdown();
}





void DDSMPI_ByTag::Send(void *buf, int count, int dest, int tag) {
    MPIMessage::MessageByTag sample;
    sample.senderRank = _rank;
    sample.recipientRank = dest;
    sample.tag = tag;
    sample.data.length(count);
    CORBA::Octet *ptr = (CORBA::Octet *)buf; 
    for (int i=0; i<count; i++)
        sample.data[i] = ptr[i];

    DDS::InstanceHandle_t instance;
    std::map<int, DDS::InstanceHandle_t>::iterator it = _writerInstances.find(tag);
    if (it != _writerInstances.end())
        instance = it->second;
    else {
        MPIMessage::MessageByTag sample;
        sample.tag = tag;
        instance = _mdw->register_instance(sample);
        _writerInstances[tag] = instance;
    }

    if (_mdw->write(sample, instance) != DDS::RETCODE_OK)
        throw new DDSException(_rank, "DDSMPI_ByTag::Send(): _mdw->write() failed");
}



void DDSMPI_ByTag::WaitOnDataAvailable(int whichReader) {
    DDS::WaitSet_var ws;
    std::map<int, DDS::WaitSet_var>::iterator it = _waitSets.find(whichReader);
    if (it != _waitSets.end())
        ws = it->second;
    else {
        ws = new DDS::WaitSet;
        DDS::StatusCondition_var dataAvailableStatus = _mdr[whichReader]->get_statuscondition();
        dataAvailableStatus->set_enabled_statuses(DDS::DATA_AVAILABLE_STATUS);
        ws->attach_condition(dataAvailableStatus);
        _statusConditions[whichReader] = dataAvailableStatus;
        _waitSets[whichReader] = ws;
    }

    DDS::Duration_t infinite = {DDS::DURATION_INFINITE_SEC, DDS::DURATION_INFINITE_NSEC};
    DDS::ConditionSeq conditions;
    conditions.length(0);
    if (DDS::RETCODE_OK != ws->wait(conditions, infinite)) 
        throw DDSException(_rank, "DDSMPI_ByTag::WaitForData(): dataAvailable->wait() failed");
}


void DDSMPI_ByTag::Take(int tag, int whichReader, MPIMessage::MessageByTagSeq &messageSeq) {
    DDS::SampleInfoSeq infoSeq;

    while (true) {  // loop until get a valid data sample
        if (tag == DDSMPI_ANY_TAG) {
            // any tag - wait for data to arrive, then take()
            WaitOnDataAvailable(whichReader);

            if (_mdr[whichReader]->take(messageSeq, infoSeq, 1, DDS::ANY_SAMPLE_STATE, DDS::ANY_VIEW_STATE, DDS::ANY_INSTANCE_STATE) != DDS::RETCODE_OK)
                throw DDSException(_rank, "DDSMPI_ByTag::Take(): r->take() should have returned data");
        }
        else {
            // specific tag - look to see if an instance handle is already known for it

            DDS::InstanceHandle_t instance;
            std::map<int, DDS::InstanceHandle_t> &instancesByReader = _readerInstances[whichReader];
            std::map<int, DDS::InstanceHandle_t>::iterator it = instancesByReader.find(tag);
            if (it!=instancesByReader.end()) {
                // have the instance handle - wait for any data on the reader, loop until data for the specific instance arrives
                instance = it->second;

                while (true) {
                    WaitOnDataAvailable(whichReader);

                    DDS::ReturnCode_t ret = _mdr[whichReader]->take_instance(messageSeq, infoSeq, 1, instance, DDS::ANY_SAMPLE_STATE, DDS::ANY_VIEW_STATE, DDS::ANY_INSTANCE_STATE);
                    if (ret == DDS::RETCODE_OK)
                        break;  // have data of the correct instance
                    if (ret == DDS::RETCODE_NO_DATA)
                        continue;  // no data for that instance, so loop to try again
                    // some other error occurred, so fail
                    throw DDSException(_rank, "DDSMPI_ByTag::Take(): r->take_instance() failed");
                }
            }
            else {
                // don't have the instance handle - use a querycondition to wait for data on the instance to arrive, and remember the instance handle for the future
                DDS::WaitSet_var ws = new DDS::WaitSet;
                std::ostringstream q;
                q << "tag = " << tag;
                std::string query = q.str();
                DDS::ReadCondition_var qc = _mdr[whichReader]->create_querycondition(DDS::ANY_SAMPLE_STATE,
                    DDS::ANY_VIEW_STATE, DDS::ALIVE_INSTANCE_STATE, query.c_str(), DDS::StringSeq());
                if (!qc.in()) 
                    throw DDSException(_rank, "DDSMPI_ByTag::WaitForData(): _mdr[whichReader]->create_querycondition() failed");
                ws->attach_condition(qc);

                DDS::Duration_t infinite = {DDS::DURATION_INFINITE_SEC, DDS::DURATION_INFINITE_NSEC};
                DDS::ConditionSeq conditions;
                conditions.length(0);
                if (DDS::RETCODE_OK != ws->wait(conditions, infinite)) 
                    throw DDSException(_rank, "DDSMPI_ByTag::WaitForData(): dataAvailable->wait() failed");

                MPIMessage::MessageByTag sample;
                sample.tag = tag;
                instance = _mdr[whichReader]->lookup_instance(sample);
                instancesByReader[tag] = instance;

                if (_mdr[whichReader]->take_instance(messageSeq, infoSeq, 1, instance, DDS::ANY_SAMPLE_STATE, DDS::ANY_VIEW_STATE, DDS::ANY_INSTANCE_STATE) != DDS::RETCODE_OK)
                    throw DDSException(_rank, "DDSMPI_ByTag::Take(): r->take_instance() failed");
            }
        }

        if (infoSeq[0].valid_data)
            break;  // exit the loop if the data was valid, else try again
    }
}


void DDSMPI_ByTag::Recv(void *buf, int count, int source, int tag, int &actualSource, int &actualTag) {
    MPIMessage::MessageByTagSeq messageSeq;
    int whichReader = (source == DDSMPI_ANY_SOURCE)?_size:source;
    Take(tag, whichReader, messageSeq);

    actualSource = messageSeq[0].senderRank;
    actualTag = messageSeq[0].tag;
    CORBA::Octet *ptr = (CORBA::Octet *)buf;
    for (int i=0; i<count; i++)
        ptr[i] = messageSeq[0].data[i];

    // take the duplicated message from the other reader
    MPIMessage::MessageByTagSeq discardSeq;
    int otherReader = (source == DDSMPI_ANY_SOURCE)?actualSource:_size;
    Take(tag, otherReader, discardSeq);
}



