#ifndef __FUTURE_H__
#define __FUTURE_H__

#include <boost/bind.hpp>
#include <boost/function.hpp>
#include <boost/shared_ptr.hpp>
#include "../IDL/MyInterfaceS.h"


// The result or exception returned by the AMI call
template <typename T>
class FutureData {
	ACE_Thread_Mutex lock_;
    T result_;
    ::Messaging::ExceptionHolder_var excep_holder_;
    bool haveReply_;
public:
    FutureData() : haveReply_(false) {}

	void SetResult(T result) {
        result_ = result;
        ACE_Guard<ACE_Thread_Mutex> guard(lock_);
        haveReply_ = true;
    }

	void SetException(::Messaging::ExceptionHolder *excep_holder) {
		excep_holder->_add_ref();
        excep_holder_ = excep_holder;
		ACE_Guard<ACE_Thread_Mutex> guard(lock_);
        haveReply_ = true;
    }

	bool HaveReply() {
        ACE_Guard<ACE_Thread_Mutex> guard(lock_);
	    return haveReply_;
    }

    // lock not needed - will only be called from the main thread
    T GetResult() const { return result_; }
    ::Messaging::ExceptionHolder_var GetException() const { return excep_holder_; };
};



// Management of the AMI call - Get() blocks until the call completes, returning either
// the result from the call or throwing an exception as returned from the call
template <typename T>
class Future {
	boost::shared_ptr<FutureData<T> > data_;
    CORBA::ORB_var orb_;
public:
    Future(CORBA::ORB_ptr orb, boost::function<void()> f, boost::shared_ptr<FutureData<T> > data)  : 
      orb_(CORBA::ORB::_duplicate(orb)), data_(data) {
        f();  // move to Get() if don't want to do work until the result is needed
    }

    Future(const Future &other) {
        orb_ = CORBA::ORB::_duplicate(other.orb_);
	    data_ = other.data_;
    }

    Future<T> &operator=(const Future &other) {
        if (this != &other) {
            orb_ = CORBA::ORB::_duplicate(other.orb_);
	        data_ = other.data_;
        }
        return *this;
    }

    // explicit Get()
    T Get() {
        if (!data_->HaveReply())
            Wait();
        if (data_->GetException())
            data_->GetException()->raise_exception();
        return data_->GetResult();
    }

    // implicit Get()
    operator T() { return Get(); }

    void Wait() {
        while (!data_->HaveReply()) {
            if (orb_->work_pending()) 
                orb_->perform_work(); 
        }
    }
};



// Generator for the servant to handle the invocation
template <typename SERVANT_TYPE, typename HANDLER, typename METHOD_RETURN>
HANDLER * CreateReplyHandler(PortableServer::POA_ptr poa,  boost::shared_ptr<FutureData<METHOD_RETURN> > data) {
	PortableServer::Servant_var<POA_Messaging::ReplyHandler> replyHandler_servant = new SERVANT_TYPE(data);
    PortableServer::ObjectId_var oid = poa->activate_object(replyHandler_servant.in());
    CORBA::Object_var handler_obj = poa->id_to_reference(oid.in());
    return HANDLER::_narrow(handler_obj.in());
}


// templatized way to build a Future that calls a method with 2 parameters - can make more like this for different numbers of parameters
template <typename METHOD_RETURN, typename SERVANT_TYPE, typename HANDLER, typename P1, typename P2, typename INTERFACE>
Future<METHOD_RETURN> CreateFuture(CORBA::ORB_ptr orb, PortableServer::POA_ptr poa, INTERFACE *myInterface, void (INTERFACE::*method)(HANDLER *, P1, P2), P1 param1, P2 param2) {
    // create a Future object with the supplied method as its function
    boost::shared_ptr<FutureData<METHOD_RETURN> > data(new FutureData<METHOD_RETURN>);
    return Future<METHOD_RETURN>(
        orb, 
        boost::bind(method, myInterface, 
            CreateReplyHandler<SERVANT_TYPE, HANDLER>(poa, data), 
            param1, param2), 
        data);
}


#endif
