#include "started_pch.h"
#include "../IDL/MyInterfaceC.h"
#include <iostream>
#include <sstream>
#include <vector>
#include "../Util/Util.h"
#include "Future.h"

// ********** IDL-SPECIFIC (MyInterface) STUFF **********


// A base class to make writing method-specific subclasses simpler
template <typename T>
class InterfaceHandlerBase : public virtual POA_MyModule::AMI_MyInterfaceHandler {
	protected:
	boost::shared_ptr<FutureData<T> > data_;
public:
    InterfaceHandlerBase(boost::shared_ptr<FutureData<T> > data) : data_(data) {}
	boost::shared_ptr<FutureData<T> > GetData() const { return data_; }

    virtual void DoWork(CORBA::Long ami_return_val) {}
    virtual void DoWork_excep(::Messaging::ExceptionHolder* excep_holder) {}

    virtual void DoMoreWork(const char * ami_return_val, CORBA::Long val) {}
    virtual void DoMoreWork_excep(::Messaging::ExceptionHolder* excep_holder) {}
};


// Handle the DoWork() method
class DoWorkHandler : public virtual InterfaceHandlerBase<CORBA::Long> {
public:
    DoWorkHandler(boost::shared_ptr<FutureData<CORBA::Long> > data) : InterfaceHandlerBase<CORBA::Long>(data) {}
    virtual void DoWork(CORBA::Long ami_return_val) { data_->SetResult(ami_return_val); }
    virtual void DoWork_excep(::Messaging::ExceptionHolder* excep_holder) { data_->SetException(excep_holder); }
};


// Handle the DoMoreWork method
class DoMoreWorkHandler : public virtual InterfaceHandlerBase<std::pair<std::string, CORBA::Long> > {
public:
    DoMoreWorkHandler(boost::shared_ptr<FutureData<std::pair<std::string, CORBA::Long> > > data) : InterfaceHandlerBase<std::pair<std::string, CORBA::Long> >(data) {}
    virtual void DoMoreWork(const char * ami_return_val, CORBA::Long val) { data_->SetResult(std::pair<std::string, CORBA::Long>(ami_return_val, val)); }
    virtual void DoMoreWork_excep(::Messaging::ExceptionHolder* excep_holder) { data_->SetException(excep_holder); }
};



// ********** APPLICATION **********


void DoWorkTest(CORBA::ORB_ptr orb, PortableServer::POA_ptr poa, int num_servers) {
	try {
		std::cout << "DoWorkTest" << std::endl;

        // Call methods
		std::vector<Future<CORBA::Long> > futures;
		for (int i=0; i<num_servers; i++) {
			MyModule::MyInterface_var myInterface = GetReference(orb, i);
			futures.push_back(CreateFuture<CORBA::Long, DoWorkHandler>(orb, poa, myInterface.in(), &MyModule::MyInterface::sendc_DoWork, 1000, false));
		}

		// Loop over Future objects and display results
		for (std::vector<Future<CORBA::Long> >::iterator it = futures.begin(); it != futures.end(); it++) 
			std::cout << "Rank: " << *it << std::endl;
	}
    catch (MyModule::MyException& ex) {
        std::cerr << "MyException exception: " << ex.what << std::endl;
    }
    catch (CORBA::Exception& ex) {
        std::cerr << "CORBA exception: " << ex << std::endl;
    }
}


void DoWorkExceptionTest(CORBA::ORB_ptr orb, PortableServer::POA_ptr poa, int num_servers) {
	try {		
		std::cout << "DoWorkExceptionTest" << std::endl;

        // Call methiods
		std::vector<Future<CORBA::Long> > futures;
		for (int i=0; i<num_servers; i++) {
			MyModule::MyInterface_var myInterface = GetReference(orb, i);
			futures.push_back(CreateFuture<CORBA::Long, DoWorkHandler>(orb, poa, myInterface.in(), &MyModule::MyInterface::sendc_DoWork, 1000, true));
		}

		// Loop over Future objects and display results
		for (std::vector<Future<CORBA::Long> >::iterator it = futures.begin(); it != futures.end(); it++) 
			std::cout << "Rank: " << *it << std::endl;
	}
    catch (MyModule::MyException& ex) {
        std::cerr << "MyException exception: " << ex.what << std::endl;
    }
    catch (CORBA::Exception& ex) {
        std::cerr << "CORBA exception: " << ex << std::endl;
    }
}


void DoMoreWorkTest(CORBA::ORB_ptr orb, PortableServer::POA_ptr poa, int num_servers) {
	try {
		std::cout << "DoMoreWorkTest" << std::endl;

        // Call methods
		std::vector<Future<std::pair<std::string, CORBA::Long> > > futures;
		for (int i=0; i<num_servers; i++) {
			MyModule::MyInterface_var myInterface = GetReference(orb, i);
			futures.push_back(CreateFuture<std::pair<std::string, CORBA::Long>, DoMoreWorkHandler>(orb, poa, myInterface.in(), &MyModule::MyInterface::sendc_DoMoreWork, 1000, false));
		}

		// Loop over Future objects and display results
		for (std::vector<Future<std::pair<std::string, CORBA::Long> > >::iterator it = futures.begin(); it != futures.end(); it++) {
			std::pair<std::string, CORBA::Long> r = *it;
			std::cout << "Rank: " << r.first << " " << r.second << std::endl;
		}
	}
    catch (MyModule::MyException& ex) {
        std::cerr << "MyException exception: " << ex.what << std::endl;
    }
    catch (CORBA::Exception& ex) {
        std::cerr << "CORBA exception: " << ex << std::endl;
    }
}



Future<CORBA::Long> f(CORBA::ORB_ptr orb, PortableServer::POA_ptr poa, int i, int delay_ms) {
    return CreateFuture<CORBA::Long, DoWorkHandler>(orb, poa, GetReference(orb, i).in(), &MyModule::MyInterface::sendc_DoWork, delay_ms, false);
}

void ChainTest(CORBA::ORB_ptr orb, PortableServer::POA_ptr poa, int num_servers) {
    if (num_servers < 5)
        return;

	try {
		std::cout << "ChainTest" << std::endl;

        // Server 4's DoWork() returns the value 4 - delay the nested call for 4 seconds
        // f(1) runs asynchronously with f(2, f(4)) - f(1) takes 3 sec, f(4) runs in 1 sec, then f(2) can start for 4 sec, so f(2, f(4)) runs in 5 sec
        // 3 sec simultaneously with 5 sec means this takes 5 sec total 
        // CORBA::Long result = f(poa, 1, 3000) + f(poa, 2, f(poa, 4, 1000) *1000); // this causes sequential start of futures (since + causes implicit Get()), so 8 sec

        Future<CORBA::Long> f1 = f(orb, poa, 1, 3000);
        Future<CORBA::Long> f2 = f(orb, poa, 2, f(orb, poa, 4, 1000) *1000);
        CORBA::Long result = f1 + f2;

        std::cout << "Result = " << result << std::endl;
	}
    catch (MyModule::MyException& ex) {
        std::cerr << "MyException exception: " << ex.what << std::endl;
    }
    catch (CORBA::Exception& ex) {
        std::cerr << "CORBA exception: " << ex << std::endl;
    }
}





// Test AMI methods.  Create num_servers server processes, connect to each one, and invoke a method.
// Then, loop over the results, displaying each value, demonstrating that the calls were made in parallel.
int main(int argc, char* argv[])
{
    try {
		// initialize
        CORBA::ORB_var orb = CORBA::ORB_init(argc, argv);
        int num_servers = atoi(argv[1]);  // number of running servers is passed on the command line

        // be a server to process AMI callbacks
        CORBA::Object_var obj = orb->resolve_initial_references("RootPOA");
        PortableServer::POA_var poa = PortableServer::POA::_narrow(obj.in());
        PortableServer::POAManager_var mgr = poa->the_POAManager();
        mgr->activate();

		// run tests
		RunTest(DoWorkTest, orb, poa, num_servers);
		RunTest(DoWorkExceptionTest, orb, poa, num_servers);
		RunTest(DoMoreWorkTest, orb, poa, num_servers);
        RunTest(ChainTest, orb, poa, num_servers);

		// clean up
        orb->destroy();
    }
    catch (MyModule::MyException& ex) {
        std::cerr << "MyException exception: " << ex.what << std::endl;
    }
    catch (CORBA::Exception& ex) {
        std::cerr << "CORBA exception: " << ex << std::endl;
    }

    return 0;
}

