#include "../SensorIDL/SensorTypeSupportImpl.h"
#include "../SensorIDL/DDSException.h"
#include <dds/DCPS/Marked_Default_Qos.h>
#include <dds/DCPS/Service_Participant.h>
#include <vcclr.h>
#include <msclr/marshal.h>

class SensorEventPublisherImpl;

public ref class SensorEventPublisher {
	SensorEventPublisherImpl *_impl;
public:
	event System::EventHandler<SensorEventLib::SensorEventArgs^> ^SensorEvent;

	SensorEventPublisher(array<System::String^>^ args);
	~SensorEventPublisher();
	bool Run();
	void Stop();
	void Publish(System::String^ sensorID, System::DateTime date, double reading);
};


class DataReaderListenerImpl
	: public virtual OpenDDS::DCPS::LocalObject<DDS::DataReaderListener> {
	gcroot<SensorEventPublisher^> _sensorEventPublisher;
public:
	DataReaderListenerImpl(SensorEventPublisher ^sensorEventPublisher) : _sensorEventPublisher(sensorEventPublisher) {}

	virtual void on_requested_deadline_missed(
		DDS::DataReader_ptr reader,
		const DDS::RequestedDeadlineMissedStatus& status) {}

	virtual void on_requested_incompatible_qos(
		DDS::DataReader_ptr reader,
		const DDS::RequestedIncompatibleQosStatus& status) {}

	virtual void on_sample_rejected(
		DDS::DataReader_ptr reader,
		const DDS::SampleRejectedStatus& status) {}

	virtual void on_liveliness_changed(
		DDS::DataReader_ptr reader,
		const DDS::LivelinessChangedStatus& status) {}

	virtual void on_data_available(
		DDS::DataReader_ptr reader);

	virtual void on_subscription_matched(
		DDS::DataReader_ptr reader,
		const DDS::SubscriptionMatchedStatus& status) {}

	virtual void on_sample_lost(
		DDS::DataReader_ptr reader,
		const DDS::SampleLostStatus& status) {}
};


void DataReaderListenerImpl::on_data_available(DDS::DataReader_ptr reader) {
	try {
		DDS::SampleInfo info;

		Sensor::SensorDataDataReader_var dr =
			Sensor::SensorDataDataReader::_narrow(reader);

		if (dr) {
			Sensor::SensorData sample;
			DDS::ReturnCode_t error = dr->take_next_sample(sample, info);
			if ((error == DDS::RETCODE_OK) && info.valid_data) {
				// post the sample as an event	- date is in seconds from midnight 1/1/1970 UTC so use that as an offset 
				// std::cout << "Received: " << sample.sensorID.in() << " " << sample.date << " " << sample.reading << std::endl;

				_sensorEventPublisher->Publish(
					gcnew System::String(sample.sensorID.in()),
					System::DateTime(1970, 1, 1, 0, 0, 0, System::DateTimeKind::Utc) + System::TimeSpan::FromSeconds(sample.date),
					sample.reading
					);
			}
		}
	}
	catch (System::Exception ^e) {
		System::Console::WriteLine(e);
	}
}


class SensorEventPublisherImpl {
	DDS::DomainParticipantFactory_var _dpf;
	DDS::DomainParticipant_var _dp;

public:
	SensorEventPublisherImpl(array<System::String^>^ args) {
		msclr::interop::marshal_context context;
		int argc = args->Length + 1;
		char **argv = new char *[argc];
		argv[0] = "";  // don't need the program name (can't be NULL though, else ACE_Arg_Shifter fails)
		for (int i = 0; i < args->Length; i++)
			argv[i + 1] = strdup(context.marshal_as<const char*>(args[i]));

		ACE::init();  // this is necessary, else get access violations on mutex creation (guard) later

		_dpf = TheParticipantFactoryWithArgs(argc, argv);
	}

	bool Run(SensorEventPublisher ^sensorEventPublisher) {
		try {
			_dp = _dpf->create_participant(42, PARTICIPANT_QOS_DEFAULT, 0, OpenDDS::DCPS::DEFAULT_STATUS_MASK);
			if (0 == _dp)
				throw DDSException("create_participant() failed");

			Sensor::SensorDataTypeSupport_var ts = new Sensor::SensorDataTypeSupportImpl();
			if (ts->register_type(_dp, "") != DDS::RETCODE_OK)
				throw DDSException("reigster_type() failed");

			DDS::Topic_var topic = _dp->create_topic("Temperature", ts->get_type_name(), TOPIC_QOS_DEFAULT, 0, OpenDDS::DCPS::DEFAULT_STATUS_MASK);
			if (0 == topic)
				throw DDSException("create_topic() failed");

			DDS::Subscriber_var sub = _dp->create_subscriber(SUBSCRIBER_QOS_DEFAULT, 0, OpenDDS::DCPS::DEFAULT_STATUS_MASK);
			if (0 == sub)
				throw DDSException("create_subscriber() failed");

			DDS::DataReaderQos dr_qos;
			sub->get_default_datareader_qos(dr_qos);
			DDS::DataReaderListener_var listener(new DataReaderListenerImpl(sensorEventPublisher));
			DDS::DataReader_var dr = sub->create_datareader(topic, dr_qos, listener, OpenDDS::DCPS::DEFAULT_STATUS_MASK);
			if (0 == dr)
				throw DDSException("create_datareader() failed");

			Sensor::SensorDataDataReader_var sdr = Sensor::SensorDataDataReader::_narrow(dr);
			if (0 == sdr)
				throw DDSException("reader _narrow() failed");
		}
		catch (const std::exception &e) {
			std::cout << "SensorSubscriber exception: " << e.what() << std::endl;
			return false;
		}
		catch (const CORBA::Exception &e) {
			e._tao_print_exception("SensorSubscriber exception: ");
			return false;
		}
		catch (System::Exception ^e) {
			System::Console::WriteLine(e);
		}

		return true;
	}

	void Stop() {
		if (0 != _dp) 
			_dp->delete_contained_entities();
		if (0 != _dpf)
			_dpf->delete_participant(_dp);

		TheServiceParticipant->shutdown();

		ACE::fini();
	}

};



SensorEventPublisher::SensorEventPublisher(array<System::String^>^ args) : _impl(new SensorEventPublisherImpl(args)) {}
SensorEventPublisher::~SensorEventPublisher() {
	delete _impl;
}
bool SensorEventPublisher::Run() {
	return _impl->Run(this);
}
void SensorEventPublisher::Stop() {
	_impl->Stop();
}

void SensorEventPublisher::Publish(System::String^ sensorID, System::DateTime date, double reading) {
	SensorEvent(this, gcnew SensorEventLib::SensorEventArgs(sensorID, date, reading));
}