package oci.xml;

import javax.xml.parsers.*;
import javax.xml.transform.*;
import javax.xml.transform.dom.*;
import javax.xml.transform.stream.*;
import org.apache.log4j.*;
import org.w3c.dom.*;

/**
 * This utility can be used to create new DOM Document objects, add nodes to
 * Document objects, as well as read and write Document objects to Files.
 * @author  Mario Aquino (OCI)
 */
public class DOMUtil {
    private static Logger log = Logger.getLogger(DOMUtil.class);
    
    /** Creates new DOMUtil */
    private DOMUtil() {
    }
    
    /**
     * This is a factory method for Document objects.  It returns a new Document
     * object.
     * @return A new empty Document object.
     */ 
    public static Document newDocument() {
        Document doc = null;
        DocumentBuilderFactory dbf = DocumentBuilderFactory.newInstance();
        try {
            DocumentBuilder db = dbf.newDocumentBuilder();
            doc = db.newDocument();
        } catch (ParserConfigurationException pce) {
            log.warn("New Document could not be created.", pce);
        }
        return doc;
    }
    
    /**
     * This method adds a child Element to the passed in Node.
     * @param node The Node to which a child Element should be added.
     * @param tagname The tagname of the new child Element
     * @return A reference to the new child Element
     */    
    public static Element addChildElement(Node node, String tagname) {
        Element child = null;
        Document doc = null;
        if (node instanceof Document) {
            doc = (Document)node;
        } else {
            doc = node.getOwnerDocument();
        }
        child = doc.createElement(tagname);
        node.appendChild(child);
        return child;
    }
    
    /**
     * This method adds a child Element to the passed in Node.  It also appends
     * a text node to that new child.
     * @param node The Node to which a child Element should be added.
     * @param tagname The tagname of the new child element
     * @param text The text for a Text child to the new child Element to the Node.
     * @return A reference to the new child Element.
     */    
    public static Element addChildElement(Node node, String tagname, String text) {
        Element child = addChildElement(node, tagname);
        Text textnode = child.getOwnerDocument().createTextNode(text);
        child.appendChild(textnode);
        return child;
    }
    
    /**
     * This method writes out the passed in Document object to the supplied
     * OutputStream, using the supplied integer as an indentation value.
     * @param doc A Document object that should be written out to the OutputStream.
     * @param out The OutputStream that will be used to write the supplied Document object
     * out to.
     * @param indent The indentation value to be used in formatting the output to the stream
     */    
    public static void writeXML(Document doc, java.io.OutputStream out, int indent) {
        StreamResult sr = new StreamResult(out);
        DOMSource ds = new DOMSource(doc);
        try {
            TransformerFactory tff = TransformerFactory.newInstance();
            Transformer tf = tff.newTransformer();
            tf.setOutputProperty(OutputKeys.INDENT, "yes");
            tf.transform(ds,  sr);
        } catch (TransformerException te) {
            log.warn("XML Output could not be written", te);
        }
    }
    
    /**
     * This method attempts to read a Document object from the supplied InputStream.
     * If no Document object can be found in the stream, this method returns
     * null.
     * @param in The InputStream from which a Document object should be read.
     * @return A reference to a Document object that has been read from the InputStream.
     */    
    public static Document readXML(java.io.InputStream in) {
        Document doc = null;
        DocumentBuilderFactory dbf = DocumentBuilderFactory.newInstance();
        try {
            DocumentBuilder db = dbf.newDocumentBuilder();
            doc = db.parse(in);
        } catch (Exception e) {
            log.warn("Exception thrown while trying to read XML Document.", e);
        }
        return doc;
    }
    
}