package oci.db;

import java.io.*;
import java.sql.*;
import oci.xml.*;
import org.apache.log4j.Logger;
import org.w3c.dom.*;

/**
 * This class can be used to query a database and return results in a <code>
 * ResultSet</code> or as a DOM <code>Document</code>.
 * @author  Mario Aquino (OCI)
 */
public class DBHelper {
    private static Logger log = Logger.getLogger(DBHelper.class);
    private Connection conn;

    /** Creates a new instance of DBHelper */
    private DBHelper() {
        try {
            initialize();
        } catch (Exception e) {
            //quietly log any exception
            log.warn("Exception thrown in constructor", e);
        }
    }

    private void initialize() throws DBHelperException {
        if (conn != null)
            return;
        try {
            conn = DBConnMgr.getInstance().getConnection();
        } catch (SQLException sqle) {
            throw new DBHelperException("An error occurred while initializing the connection resource",
            sqle);
        }
    }

    /**
     * Factory method.
     * @return A new instance of DBHelper.
     */
    public static DBHelper getDBHelper() {
        return new DBHelper();
    }

    /**
     * This method returns a <code>Resultset</code> based on the query string
     * it is passed.
     * @param sqlQuery An SQL query string.
     * @throws DBHelperException An exception is thrown if the query cannot be
     * successfully executed by the database.
     * @return A <code>ResultSet</code> containing data from the query string
     * passed into the method.
     */
    public ResultSet getResultSet(String sqlQuery) throws DBHelperException {
        log.debug(sqlQuery);
        initialize();
        ResultSet rs = null;
        try {
            Statement stmt = conn.createStatement();
            rs = stmt.executeQuery(sqlQuery);
        } catch (SQLException sqle) {
            throw new DBHelperException(
            "SQLException thrown while parsing resultset", sqle);
        }
        return rs;
    }

    /**
     * This method queries a database with the SQL string passed in and returns
     * the results from the query in a DOM document.  The root of the document
     * is a node called "ResultSet" and it has child nodes called "row", each
     * having child nodes that contain the result of query.  A sample Document
     * may look like this:
     * <P><FONT face='courier'>
     * &lt;?xml version="1.0" encoding="UTF-8"?&gt;<br/>
     * &lt;ResultSet&gt;<br/>
     *  &lt;row&gt;<br/>
     *      &lt;field1 type="integer"&gt;12&lt;/field1&gt;<br/>
     *      &lt;field2 type="Varchar"&gt;Foo&lt;/field2&gt;<br/>
     *  &lt;/row&gt;<br/>
     *  &lt;row&gt;<br/>
     *      &lt;field1 type="integer"&gt;13&lt;/field1&gt;<br/>
     *      &lt;field2 type="Varchar"&gt;Bar&lt;/field2&gt;<br/>
     *  &lt;/row&gt;<br/>
     * &lt;/ResultSet&gt;
     * </FONT></P>
     * @param sqlQuery The SQL query that needs to be passed to the database.
     * @throws DBHelperException A DBHelperException may be thrown if any
     * problem occurs while the instance is retrieving data from the database
     * or while the instance is creating a DOM Document object
     * with the results from the query.
     * @return A DOM Document containing information from the result set
     * retrieved from the database query.
     */
    public Document getResultSetAsDocument(String sqlQuery)
    throws DBHelperException {

        ResultSet rs = getResultSet(sqlQuery);
        return convertRSToDocument(rs);

    }

    private Document convertRSToDocument(ResultSet rs) throws DBHelperException {
        Document doc = null;

        try {
            ResultSetMetaData rsmd = rs.getMetaData();

            //Get an array of columns from the RSMetaData
            String[] columns = new String[rsmd.getColumnCount()];
            for (int i = 0; i < columns.length; i++) {
                columns[i] = rsmd.getColumnName(i + 1);
                if (log.isDebugEnabled()) log.debug("Column name = " + columns[i]);
            }

            //Get an array of the types of each RS column from the RSMetaData
            int[] columnTypes = new int[columns.length];
            for (int i = 0; i < columnTypes.length; i++) {
                columnTypes[i] = rsmd.getColumnType(i + 1);
                if (log.isDebugEnabled()) log.debug("Column type = " +
                columnTypes[i]);
            }

            doc = DOMUtil.newDocument();
            Element root = DOMUtil.addChildElement(doc, "ResultSet");

            Element row = null;
            Element elem = null;
            Text text = null;
            ByteArrayOutputStream baos = null;
            while (rs.next()) {
                log.debug("New row in resultset");
                row = DOMUtil.addChildElement(root, "row");

                for (int i = 0; i < columns.length; i++) {
                    elem = DOMUtil.addChildElement(row, columns[i]);

                    switch (columnTypes[i]) {
                        case Types.BIGINT:
                            log.debug("BigInt element found in row");
                            long bint = rs.getLong(i + 1);
                            elem.setAttribute("type", "BigInt");
                            text = doc.createTextNode(Long.toString(bint));
                            if (log.isDebugEnabled()) log.debug("BigInt = " +
                            Long.toString(bint));
                            break;
                        case Types.VARBINARY:
                        case Types.LONGVARBINARY:
                        case Types.BINARY:
                            log.debug("Binary element found in row");
                            InputStream is = rs.getBinaryStream(i + 1);
                            baos =
                            new ByteArrayOutputStream();
                            try {
                                int byt = is.read();
                                while (byt != -1) {
                                    baos.write(byt);
                                    byt = is.read();
                                }
                            } catch (IOException ioe) {
                                log.warn("Unable to read the binary field", ioe);
                            }
                            elem.setAttribute("type", "Binary");
                            text = doc.createTextNode(baos.toString());
                            break;
                        case Types.BLOB:
                            log.debug("BLOB element found in row");
                            Blob blob = rs.getBlob(i + 1);
                            byte[] barr = blob.getBytes(1, (int)blob.length());
                            baos =
                            new ByteArrayOutputStream(barr.length);
                            baos.write(barr, 0, barr.length);
                            elem.setAttribute("type", "BLOB");
                            text = doc.createTextNode(baos.toString());
                            break;
                        case Types.BOOLEAN:
                            log.debug("Boolean element found in row");
                            boolean bool = rs.getBoolean(i + 1);
                            elem.setAttribute("type", "boolean");
                            text = doc.createTextNode(Boolean.toString(bool));
                            if (log.isDebugEnabled()) log.debug("Boolean = " +
                            Boolean.toString(bool));
                            break;
                        case Types.CLOB:
                            log.debug("CLOB element found in row");
                            Clob clob = rs.getClob(i + 1);
                            elem.setAttribute("type",  "CLOB");
                            if (clob == null) {
								//Add a null attribute to the element to represent null
								//value in the database
                                elem.setAttribute("null", "true");
                                text = doc.createTextNode("");
							} else
								text = doc.createTextNode(clob.getSubString(1L,(int)clob.length()));
                            break;
                        case Types.DATE:
                            log.debug("Date element found in row");
                            Date date = rs.getDate(i + 1);
                            elem.setAttribute("type", "Date");
                            if (date == null) {
								//Add a null attribute to the element to represent null
								//value in the database
                                elem.setAttribute("null", "true");
                                text = doc.createTextNode("");
							} else
								text = doc.createTextNode(date.toString());
                            if (log.isDebugEnabled()) log.debug("Date = " + date.toString());
                            break;
                        case Types.NUMERIC:
                        case Types.DECIMAL:
                        case Types.DOUBLE:
                            log.debug("Double element found in row");
                            double doubl = rs.getDouble(i + 1);
                            elem.setAttribute("type",  "double");
                            text = doc.createTextNode(Double.toString(doubl));
                            if (log.isDebugEnabled()) log.debug("Double = " +
                            Double.toString(doubl));
                            break;
                        case Types.REAL:
                        case Types.FLOAT:
                            log.debug("Float element found in row");
                            float fl = rs.getFloat(i + 1);
                            elem.setAttribute("type", "float");
                            text = doc.createTextNode(Float.toString(fl));
                            if (log.isDebugEnabled()) log.debug("Float = " +
                            Float.toString(fl));
                            break;
                        case Types.INTEGER:
                            log.debug("Integer element found in row");
                            int integer = rs.getInt(i + 1);
                            elem.setAttribute("type",  "integer");
                            text = doc.createTextNode(Integer.toString(integer));
                            if (log.isDebugEnabled()) log.debug("Integer = " +
                            Integer.toString(integer));
                            break;
                        case Types.NULL:
                            log.debug("Null element found in row");
                            elem.setAttribute("type", "Null");
                            text = doc.createTextNode("");
                            break;
                        case Types.TIME:
                            log.debug("Time element found in row");
                            Time time = rs.getTime(i + 1);
                            elem.setAttribute("type", "Time");
                            if (time == null) {
								//Add a null attribute to the element to represent null
								//value in the database
                                elem.setAttribute("null", "true");
                                text = doc.createTextNode("");
							} else
								text = doc.createTextNode(time.toString());
                            if (log.isDebugEnabled()) log.debug("Time = " + time.toString());
                            break;
                        case Types.TIMESTAMP:
                            log.debug("Timestamp element found in row");
                            Timestamp ts = rs.getTimestamp(i + 1);
                            elem.setAttribute("type", "Timestamp");
                            if (ts == null) {
								//Add a null attribute to the element to represent null
								//value in the database
                                elem.setAttribute("null", "true");
                                text = doc.createTextNode("");
							} else
								text = doc.createTextNode(ts.toString());
                            if (log.isDebugEnabled()) log.debug("Timestamp = " + ts.toString());
                            break;
                        case Types.SMALLINT:
                        case Types.TINYINT:
                            log.debug("TinyInt element found in row");
                            short shrt = rs.getShort(i + 1);
                            elem.setAttribute("type", "TinyInt");
                            text = doc.createTextNode(Short.toString(shrt));
                            if (log.isDebugEnabled()) log.debug("Short = " +
                            Short.toString(shrt));
                            break;
                        case Types.LONGVARCHAR:
                        case Types.VARCHAR:
                            log.debug("Varchar element found in row");
                            String str = rs.getString(i + 1);
                            elem.setAttribute("type",  "Varchar");
                            if (str == null) {
								//Add a null attribute to the element to represent null
                                //value in the database
                                elem.setAttribute("null", "true");
                                str = "";
							}
							text = (Text)doc.createCDATASection(str);
                            if (log.isDebugEnabled()) log.debug("Varchar = " +
                            str);
                            break;
                        default:
                            //log here because we didn't figure out
                            //what this thing was
                            log.warn("Unknown element type found!!!");
                            log.warn("Column = " + columns[i]);
                            elem.setAttribute("type", "Unknown");
                            text = doc.createTextNode("");
                            break;
                    }
                    elem.appendChild(text);
                }
            }

        } catch (SQLException sqle) {
            throw new DBHelperException(
            "SQLException thrown while parsing resultset", sqle);
        }

        return doc;

    }

    /**
     * This method returns a document containing metadata about the tables
     * indicated in the query.  For each table belonging to the catalog and
     * schema, the document will contain:
     *  The table's name, it's columns (name, type), it's primary keys, and it's
     * foreign keys (imported keys).
     */
    public Document getDBMetadataAsDocument(String catalog, String schemaPattern,
        String tableNamePattern, String[] types) throws DBHelperException
    {
        Document doc = DOMUtil.newDocument();
        Element root = DOMUtil.addChildElement(doc, "root");
        try {
            DatabaseMetaData dbmd = conn.getMetaData();
            ResultSet rs = dbmd.getTables(catalog, schemaPattern, tableNamePattern, types);
            Element tables = DOMUtil.addChildElement(root, "tables");
            tables.appendChild(
                doc.importNode(
                    convertRSToDocument(rs).getDocumentElement(),
                    true));
            //Our document now has all the tables in the database, for each
            //table, we need to get the rest of the metadata
            //To get the table names, we need to query the document for any
            //elements called 'TABLE_NAME'.  These came from the call we just made.
            NodeList nodes = doc.getElementsByTagName("TABLE_NAME");
            String[] tableNames = new String[nodes.getLength()];
            for (int i = 0; i < nodes.getLength(); i++) {
                tableNames[i] = nodes.item(i).getFirstChild().getNodeValue();
                log.debug("Table name = " + tableNames[i]);
            }

            for (int i = 0; i < tableNames.length; i++) {
                log.debug("Getting columns...");
                rs = dbmd.getColumns(catalog, schemaPattern, tableNames[i], null);
                addResultsToNode(rs, root, "columns");

                log.debug("Getting primarykeys...");
                rs = dbmd.getPrimaryKeys(catalog, schemaPattern, tableNames[i]);
                addResultsToNode(rs, root, "primarykeys");

                log.debug("Getting importedkeys...");
                rs = dbmd.getImportedKeys(catalog, schemaPattern, tableNames[i]);
                addResultsToNode(rs, root, "importedkeys");
            }

            //Change the DATA_TYPE elements to their SQL type names just for
            //aesthetics... They are actually returned as integers by the
            //DatabaseMetaData API
            nodes = doc.getElementsByTagName("DATA_TYPE");
            for (int i = 0; i < nodes.getLength(); i++) {
                int type = Integer.parseInt(
                            nodes.item(i).getFirstChild().getNodeValue());
                switch(type) {
                    case Types.ARRAY:
                        nodes.item(i).getFirstChild().setNodeValue("Array");
                        break;
                    case Types.BIGINT:
                        nodes.item(i).getFirstChild().setNodeValue("BigInt");
                        break;
                    case Types.BINARY:
                        nodes.item(i).getFirstChild().setNodeValue("Binary");
                        break;
                    case Types.BIT:
                        nodes.item(i).getFirstChild().setNodeValue("Bit");
                        break;
                    case Types.BLOB:
                        nodes.item(i).getFirstChild().setNodeValue("Blob");
                        break;
                    case Types.BOOLEAN:
                        nodes.item(i).getFirstChild().setNodeValue("Boolean");
                        break;
                    case Types.CHAR:
                        nodes.item(i).getFirstChild().setNodeValue("Char");
                        break;
                    case Types.CLOB:
                        nodes.item(i).getFirstChild().setNodeValue("Clob");
                        break;
                    case Types.DATALINK:
                        nodes.item(i).getFirstChild().setNodeValue("Datalink");
                        break;
                    case Types.DATE:
                        nodes.item(i).getFirstChild().setNodeValue("Date");
                        break;
                    case Types.DECIMAL:
                        nodes.item(i).getFirstChild().setNodeValue("Decimal");
                        break;
                    case Types.DISTINCT:
                        nodes.item(i).getFirstChild().setNodeValue("Distinct");
                        break;
                    case Types.DOUBLE:
                        nodes.item(i).getFirstChild().setNodeValue("Double");
                        break;
                    case Types.FLOAT:
                        nodes.item(i).getFirstChild().setNodeValue("Float");
                        break;
                    case Types.INTEGER:
                        nodes.item(i).getFirstChild().setNodeValue("Integer");
                        break;
                    case Types.JAVA_OBJECT:
                        nodes.item(i).getFirstChild().setNodeValue("Java_Object");
                        break;
                    case Types.LONGVARBINARY:
                        nodes.item(i).getFirstChild().setNodeValue("LongVarbinary");
                        break;
                    case Types.LONGVARCHAR:
                        nodes.item(i).getFirstChild().setNodeValue("LongVarchar");
                        break;
                    case Types.NULL:
                        nodes.item(i).getFirstChild().setNodeValue("Null");
                        break;
                    case Types.NUMERIC:
                        nodes.item(i).getFirstChild().setNodeValue("Numeric");
                        break;
                    case Types.OTHER:
                        nodes.item(i).getFirstChild().setNodeValue("Other");
                        break;
                    case Types.REAL:
                        nodes.item(i).getFirstChild().setNodeValue("Real");
                        break;
                    case Types.REF:
                        nodes.item(i).getFirstChild().setNodeValue("Ref");
                        break;
                    case Types.SMALLINT:
                        nodes.item(i).getFirstChild().setNodeValue("SmallInt");
                        break;
                    case Types.STRUCT:
                        nodes.item(i).getFirstChild().setNodeValue("Struct");
                        break;
                    case Types.TIME:
                        nodes.item(i).getFirstChild().setNodeValue("Time");
                        break;
                    case Types.TIMESTAMP:
                        nodes.item(i).getFirstChild().setNodeValue("Timestamp");
                        break;
                    case Types.TINYINT:
                        nodes.item(i).getFirstChild().setNodeValue("TinyInt");
                        break;
                    case Types.VARBINARY:
                        nodes.item(i).getFirstChild().setNodeValue("Varbinary");
                        break;
                    case Types.VARCHAR:
                        nodes.item(i).getFirstChild().setNodeValue("Varchar");
                        break;
                    default:
                        nodes.item(i).getFirstChild().setNodeValue("Unknown");
                        break;
                }
            }

        } catch (SQLException sqle) {
            throw new DBHelperException(
            "SQLException thrown while retrieving DatabaseMetaData", sqle);
        } catch (DOMException de) {
            throw new DBHelperException(
            "DOMException thrown while retrieving DatabaseMetaData", de);
        }

        return doc;
    }

    private void addResultsToNode(ResultSet rs, Node node, String elementName)
        throws DBHelperException
    {
        Element elem = DOMUtil.addChildElement(node, elementName);
        elem.appendChild(
            node.getOwnerDocument().importNode(
                convertRSToDocument(rs).getDocumentElement(),
                true));
    }

    /**
     * This method frees any resources that may be held by the instance of
     * the DBHelper.  This method should be called after the last use of the
     * DBHelper instance.
     */
    public void freeResources() {
        try {
            if ((conn != null)&&(!conn.isClosed()))
                DBConnMgr.getInstance().freeConnection(conn);
        } catch (Exception e) {
            log.warn("There was an error freeing the resources", e);
        }
    }
}
