package com.ociweb.jnb.ex3;

import junit.framework.TestCase;

import java.sql.Connection;
import java.sql.SQLException;
import java.util.Properties;

public class TestTransactionManagerImpl extends TestCase {
    private TransactionManagerImpl manager;
    private MockPropertiesManager mockPropertiesManager;
    private MockConnection mockConnection;
    private MockConnectionFactory mockConnectionFactory;
    public static final String DB_DRIVER = "TestDBDriver";
    public static final String DB_URL = "URL";
    public static final String USERNAME = "Mario";
    public static final String PASSWORD = "secret";

    public void setUp() {
        mockConnection = new MockConnection();
        mockConnectionFactory = new MockConnectionFactory();
        mockConnectionFactory.createConnectionReturn = mockConnection;
        Properties props = new Properties();
        props.setProperty("DB_DRIVER", DB_DRIVER);
        props.setProperty("DB_URL", DB_URL);
        props.setProperty("USERNAME", USERNAME);
        props.setProperty("PASSWORD", PASSWORD);
        mockPropertiesManager = new MockPropertiesManager();
        mockPropertiesManager.loadPropertiesReturn = props;
        GlobalPropertiesManager.set(mockPropertiesManager);
    }

    public TestTransactionManagerImpl(String name) {
        super(name);
    }

    public void testTransactionManagerInitialization() {
        manager = new TransactionManagerImpl(mockConnectionFactory);

        //Test that properties were loaded and used correctly
        assertTrue(mockPropertiesManager.loadPropertiesCalled);
        assertEquals("TransactionManager.properties", mockPropertiesManager.loadPropertiesPropertiesFileName);
        assertEquals(DB_DRIVER, mockConnectionFactory.createConnectionDatabaseDriver);
        assertEquals(DB_URL, mockConnectionFactory.createConnectionDatabaseURL);
        assertEquals(USERNAME, mockConnectionFactory.createConnectionUsername);
        assertEquals(PASSWORD, mockConnectionFactory.createConnectionPassword);

        assertTrue(mockConnectionFactory.createConnectionCalled);
        assertTrue(mockConnection.setAutoCommitCalled);
        assertEquals(false, mockConnection.setAutoCommitAutoCommit);
    }

    public void testCommitSuccess() throws Exception {
        manager = new TransactionManagerImpl(mockConnectionFactory);
        manager.commit();
        assertTrue(mockConnection.commitCalled);
    }

    public void testCommitFailure() throws  Exception {
        mockConnection.commitException = new SQLException("Could not commit!");
        manager = new TransactionManagerImpl(mockConnectionFactory);
        try {
            manager.commit();
            fail("The TransactionManager should wrap an SQLException in a TransactionException");
        } catch (TransactionException e) {
            assertTrue(mockConnection.commitCalled);
            assertTrue(e.getCause() instanceof SQLException);
            assertEquals("Could not commit!", e.getCause().getMessage());
        }
    }

    public void testRollbackSuccess() throws  Exception {
        manager = new TransactionManagerImpl(mockConnectionFactory);
        manager.rollback();
        assertTrue(mockConnection.rollbackCalled);
    }

    public void testRollbackFailure() throws  Exception {
        mockConnection.rollbackException = new SQLException("SQL Failure!");
        manager = new TransactionManagerImpl(mockConnectionFactory);
        try {
            manager.rollback();
            fail("The TransactionManager should wrap an SQLException in a TransactionException");
        } catch (TransactionException e) {
            assertTrue(mockConnection.rollbackCalled);
            assertTrue(e.getCause() instanceof SQLException);
            assertEquals("SQL Failure!", e.getCause().getMessage());
        }
    }

    public void testConnectionProxy() throws  Exception {
        manager = new TransactionManagerImpl(mockConnectionFactory);
        Connection proxy = manager.getConnection();
        assertFalse("The connection commit has not been called yet", mockConnection.commitCalled);
        proxy.commit();
        assertFalse("The connection commit has still not been called", mockConnection.commitCalled);
        assertFalse("The connection rollback has not been called yet", mockConnection.rollbackCalled);
        proxy.rollback();
        assertFalse("The connection rollback has still not been called", mockConnection.rollbackCalled);
        assertFalse("The connection close has not been called", mockConnection.closeCalled);
        proxy.close();
        assertTrue("The proxy caused the real connection.close() method to be called", mockConnection.closeCalled);
    }
}
