package com.ociweb.jmx.jaas.srp;

import org.jboss.security.Util;
import org.jboss.security.srp.SRPConf;
import org.jboss.security.srp.SRPVerifierStore;

import javax.naming.Context;
import javax.naming.InitialContext;
import javax.naming.NamingException;
import javax.sql.DataSource;
import java.io.IOException;
import java.math.BigInteger;
import java.security.KeyException;
import java.security.NoSuchAlgorithmException;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;

/**
 * @author Brian M. Coyner
 */
public class DatabaseVerifierStore implements SRPVerifierStore {

    private Context context;

    private String dsJndiName;
    private String passwordQuery;
    private Map users;
    private BigInteger g;
    private BigInteger N;

    public DatabaseVerifierStore(String dsJndiName, String passwordQuery) {
        this.dsJndiName = dsJndiName;
        this.passwordQuery = passwordQuery;
        this.users = Collections.synchronizedMap(new HashMap());

        try {
            Util.init();
        } catch (NoSuchAlgorithmException e) {
        }

        this.g = SRPConf.getDefaultParams().g();
        this.N = SRPConf.getDefaultParams().N();
    }

    public VerifierInfo getUserVerifier(String username)
            throws KeyException, IOException {
        VerifierInfo info = (VerifierInfo) this.users.get(username);
        if (info == null) {
            try {
                info = createVerifierInfo(username, lookupPassword(username));
                setUserVerifier(username, info);
            } catch (Exception e) {
                throw new IOException("Unable to lookup user credentials.  " +
                        e.getMessage());
            }
        }

        return info;
    }

    public void setUserVerifier(String username, VerifierInfo info) {
        Object prevObject = this.users.put(username, info);
        if (prevObject == null) {
            //should have logging here
            System.out.println("    New User Added");
        } else {
            System.out.println("    User Updated");
        }
    }

    public void addNewUser(String username, String password) throws Exception {
        try {
            setUserVerifier(username, createVerifierInfo(username, password));
        } catch (Throwable e) {
            throw new Exception(e);
        }
    }

    public void deleteUser(String username) {
        //tbd: delete user
    }

    private String lookupPassword(String username)
            throws NamingException, SQLException, KeyException {

        if (this.context == null) {
            this.context = new InitialContext();
        }

        DataSource ds = (DataSource) this.context.lookup(this.dsJndiName);
        Connection conn = ds.getConnection();
        PreparedStatement pStmt = conn.prepareStatement(this.passwordQuery);
        pStmt.setString(1, username);
        ResultSet rs = pStmt.executeQuery();
        String pw = null;
        int rowCount = 0;

        while (rs.next()) {
            pw = rs.getString(1);
            rowCount ++;
        }

        if (rowCount == 0) {
            throw new KeyException("User [" + username + "] not found");
        } else if (rowCount == 1) {
            return pw;
        } else {
            throw new SQLException("Invalid SQL Query.  " +
                                   "Multiple Rows were found.  " +
                                   "Please fix your query.");
        }
    }

    private VerifierInfo createVerifierInfo(String username, String password) {
        VerifierInfo info = new VerifierInfo();

        info.username = username;
        info.salt = "123456".getBytes();
        info.g = this.g.toByteArray();
        info.N = this.N.toByteArray();
        info.verifier = Util.calculateVerifier(username,
                password.toCharArray(),
                info.salt,
                info.N,
                info.g);

        return info;
    }
}
