package jnb;

import java.util.*;

public class Customer {
    private final String name;
    private final Map accounts = new HashMap();

    static final String CUSTOMER_NAME_NOT_NULL_MESSAGE = 
            "Customer name cannot be null or empty";

    /**
     * Constructor
     * @param name
     * @throws IllegalArgumentException thrown if the name passed in is null
     */
    public Customer(String name) {
        if (name == null || name.length() == 0) {
            throw new IllegalArgumentException(CUSTOMER_NAME_NOT_NULL_MESSAGE);
        }
        this.name = name;
    }

    /**
     * Adds the Account to the set of those belonging to the Customer
     * @param account
     */
    public void addAccount(Account account) {
        accounts.put(account.getId(), account);
    }

    /**
     * Removes the Account from the set of those belonging to the Customer
     * @param account
     */
    public void removeAccount(Account account) {
        accounts.remove(account);
    }

    /**
     * Returns the sum (in Money) of the balances of all Account objects
     * belonging to the Customer instance.  If this customer has no accounts,
     * a Money of value zero will be returned.
     * @return The cumulative balance of all Accounts held by this customer
     */
    public Money getBalance() {
        Money currentBalance = new Money("0.00");
        for(Iterator iter = accounts.values().iterator(); iter.hasNext();) {
            Account account = (Account) iter.next();
            currentBalance = currentBalance.add(account.getBalance());
        }
        return currentBalance;
    }

    public boolean equals(Object o) {
        if (this == o) return true;
        if (!(o instanceof Customer)) return false;

        final Customer customer = (Customer) o;

        if (!name.equals(customer.name)) return false;

        return true;
    }

    public int hashCode() {
        return name.hashCode();
    }

    public Collection getAccounts() {
        return Collections.unmodifiableCollection(accounts.values());
    }

    public String getName()
    {
        return name;
    }

    public String toString()
    {
        return name;
    }

    public Account getAccount(String accountID) {
        return (Account)accounts.get(accountID);
    }
}