package jnb;

public class Account {
    private final String id;
    private final Customer customer;
    private Money balance;
    private static final Money ZERO_MONEY_VALUE = new Money("0.00");

    static final String DEPOSIT_MORE_THAN_ZERO_MESSAGE =
            "Deposit amounts must be greater than zero";
    static final String WITHDRAWL_AMOUNT_MORE_THAN_BALANCE_MESSAGE = 
            "Withdrawal amounts must be less than the current balance";
    static final String CANNOT_TRANSFER_BETWEEN_DIFFERENT_CUSTOMERS_MESSAGE =
            "Cannot perform transfers between different customers";
    static final String CANNOT_TRANSER_WITHIN_THE_SAME_ACCOUNT_MESSAGE =
            "Transfers must be between two different accounts";

    public Account(String id, Customer customer, Money balance) {
        this.id = id;
        this.customer = customer;
        this.balance = balance;
    }

    public Money getBalance() {
        return balance;
    }

    public Customer getCustomer() {
        return customer;
    }

    public String getId() {
        return id;
    }

    /**
     * Adds the passed in Money amount to this Account's balance.  Deposits
     * amounts must be greater than zero.
     * @param amount
     * @throws InvalidTransactionException Thrown if the amount being deposited
     * is not greater than zero.
     */
    public void deposit(Money amount) throws InvalidTransactionException {
        if (!amount.isGreaterThan(ZERO_MONEY_VALUE)) {
            throw new InvalidTransactionException(DEPOSIT_MORE_THAN_ZERO_MESSAGE);
        }
        balance = balance.add(amount);
    }

    /**
     * Removes the amount passed in from the current balance of this Account.
     * @param amount The quantity of Money to be withdrawn from this Account.
     * @return A Money equal to the amount passed in (i.e. the amount withdrawn)
     * @throws InsufficientFundsException Thrown if the current Account balance
     * is less than the amount request for withdrawl
     */
    public Money  withdrawal(Money amount) throws InsufficientFundsException {
        if (amount.isGreaterThan(balance)) {
            throw new InsufficientFundsException(WITHDRAWL_AMOUNT_MORE_THAN_BALANCE_MESSAGE);
        }
        balance = balance.subtract(amount);
        return amount;
    }

    /**
     * Moves funds (Money) from the Account passed in to this Account.  Transfers
     * are only allowed between Accounts held by the same Customer.  Transfers
     * are only allowed between two different Account objects (i.e. transfer
     * will fail if the fromAccount is equal to this Account)
     * @param fromAccount The Account from which funds will be withdrawn (to
     * deposit into this Account)
     * @param amount The quantity of Money being transfered.
     * @throws InvalidTransactionException Thrown if the fromAccount Customer
     * object is not equal to the Customer for this Account or if the fromAccount
     * is equal to this Account (i.e. both Account objects are equal)
     * @throws InsufficientFundsException
     */
    public void transferFunds(Account fromAccount, Money amount)
            throws InvalidTransactionException, InsufficientFundsException{
        if (!customer.equals(fromAccount.customer)) {
            throw new InvalidTransactionException(CANNOT_TRANSFER_BETWEEN_DIFFERENT_CUSTOMERS_MESSAGE);
        } else if (equals(fromAccount)) {
            throw new InvalidTransactionException(CANNOT_TRANSER_WITHIN_THE_SAME_ACCOUNT_MESSAGE);
        }
        deposit(fromAccount.withdrawal(amount));
    }

    public boolean equals(Object o) {
        if (this == o) return true;
        if (!(o instanceof Account)) return false;

        final Account account = (Account) o;

        if (!customer.equals(account.customer)) return false;
        if (!id.equals(account.id)) return false;

        return true;
    }

    public int hashCode() {
        int result;
        result = id.hashCode();
        result = 29 * result + customer.hashCode();
        return result;
    }

    public String toString()
    {
        return id;
    }
}