package jnb;

import java.math.BigDecimal;

public class Money {
    private final BigDecimal value;
    static final String NEGATIVE_AMOUNT_NOT_ALLOWED_MESSAGE = 
            "Money amounts must be postive. Negative amounts are not allowed.";
    private static final BigDecimal ZERO = new BigDecimal("0.00");

    public Money(String value) throws NumberFormatException {
        this(new BigDecimal(value.replaceAll("\\$", "")));
    }

    private Money(BigDecimal value) {
        this.value = value;
        if (ZERO.equals(value)) {
            return;
        }
        BigDecimal bigger = ZERO.max(value);
        if (bigger.equals(ZERO)) {
            throw new NumberFormatException(NEGATIVE_AMOUNT_NOT_ALLOWED_MESSAGE);
        }
    }

    /**
     * Returns a new Money that is equal to the sum of the Money passed in plus
     * the value of this Money.
     * @param money
     * @return
     */
    public Money add(Money money) {
        return new Money(value.add(money.value));
    }

    /**
     * Returns a new Money that is equal to the remainder after the Money passed
     * is subtracted from the value of this Money.
     * @param money
     * @return
     * @throws NumberFormatException Thrown if the result of subtracting the
     * value of the Money passed in from the value of this Money object is a
     * negative amount.
     */
    public Money subtract(Money money) {
        return new Money(value.subtract(money.value));
    }

    /**
     * Returns true if the Money passed in is greater than the value of this
     * Money object.  Otherwise returns false.
     * @param money
     * @return
     */
    public boolean isGreaterThan(Money money) {
        if (money.value.equals(value)) {
            return false;
        }
        BigDecimal most = value.max(money.value);
        return most.equals(value);
    }

    public String toString() {
        return "$" + value.toString();
    }

    public boolean equals(Object o) {
        if (this == o) return true;
        if (!(o instanceof Money)) return false;

        final Money money = (Money) o;

        if (!value.equals(money.value)) return false;

        return true;
    }

    public int hashCode() {
        return value.hashCode();
    }
}