#include <dds/DdsDcpsInfrastructureC.h>
#include <dds/DdsDcpsSubscriptionC.h>
#include <dds/DCPS/Marked_Default_Qos.h>
#include <dds/DCPS/Service_Participant.h>
#include <dds/DCPS/WaitSet.h>
#include "../SensorIDL/SensorTypeSupportImpl.h"
#include "../SensorIDL/DDSException.h"
#include "rx.hpp"
#include "ace/OS_NS_time.h"
#include <map>

namespace rxsub = rxcpp::subjects;

class DataReaderListenerImpl
	: public virtual OpenDDS::DCPS::LocalObject < DDS::DataReaderListener > {
	rxsub::subject<rxsub::subject<Sensor::SensorData>> & _obsGroup;
	std::map<DDS::InstanceHandle_t, rxsub::subject<Sensor::SensorData>> _obsMap;
public:
	DataReaderListenerImpl(rxsub::subject<rxsub::subject<Sensor::SensorData>> &obsGroup) : _obsGroup(obsGroup) {}

	virtual void on_requested_deadline_missed(
		DDS::DataReader_ptr reader,
		const DDS::RequestedDeadlineMissedStatus& status) {}

	virtual void on_requested_incompatible_qos(
		DDS::DataReader_ptr reader,
		const DDS::RequestedIncompatibleQosStatus& status) {}

	virtual void on_sample_rejected(
		DDS::DataReader_ptr reader,
		const DDS::SampleRejectedStatus& status) { }

	virtual void on_liveliness_changed(
		DDS::DataReader_ptr reader,
		const DDS::LivelinessChangedStatus& status) {}

	virtual void on_data_available(
		DDS::DataReader_ptr reader);

	virtual void on_subscription_matched(
		DDS::DataReader_ptr reader,
		const DDS::SubscriptionMatchedStatus& status) {}

	virtual void on_sample_lost(
		DDS::DataReader_ptr reader,
		const DDS::SampleLostStatus& status) {}
};



void DataReaderListenerImpl::on_data_available(DDS::DataReader_ptr reader) {
	Sensor::SensorData sensorData;
	DDS::SampleInfo info;

	Sensor::SensorDataDataReader_var dr =
		Sensor::SensorDataDataReader::_narrow(reader);

	if (dr) {
		Sensor::SensorData sample;
		DDS::ReturnCode_t error = dr->take_next_sample(sample, info);
		if (error == DDS::RETCODE_OK) {

			// if instance is not found in the map, create a new observable for it and push it out
			auto obs = rxsub::subject<Sensor::SensorData>();
			auto obsIt = _obsMap.find(info.instance_handle);
			if (obsIt == _obsMap.end()) {
				_obsGroup.get_subscriber().on_next(obs);
				_obsMap[info.instance_handle] = obs;
			}
			else
				obs = obsIt->second;

			// update the observable
			if (info.valid_data) {
				// data is valid - call OnNext on the observable with the data
				obs.get_subscriber().on_next(sample);
			}
			else if (info.instance_state == DDS::NOT_ALIVE_DISPOSED_INSTANCE_STATE) {
				// no more data - call OnCompleted on the observable
				obs.get_subscriber().on_completed();
			}

		}
		else {
			// can't take the sample - assume we're dead and set an error state on all observables
			std::exception_ptr ex = std::make_exception_ptr(std::runtime_error("take_next_sample() failed"));
			std::for_each(_obsMap.begin(), _obsMap.end(), 
				[ex](std::pair<const DDS::InstanceHandle_t, rxsub::subject<Sensor::SensorData>>& obs){
				obs.second.get_subscriber().on_error(ex);
			});
		}
	}
}



int ACE_TMAIN(int argc, ACE_TCHAR *argv[])
{
	try {
		DDS::DomainParticipantFactory_var dpf = TheParticipantFactoryWithArgs(argc, argv);

		DDS::DomainParticipant_var dp = dpf->create_participant(42, PARTICIPANT_QOS_DEFAULT, 0, OpenDDS::DCPS::DEFAULT_STATUS_MASK);
		if (0 == dp)
			throw DDSException("create_participant() failed");

		Sensor::SensorDataTypeSupport_var ts = new Sensor::SensorDataTypeSupportImpl();
		if (ts->register_type(dp, "") != DDS::RETCODE_OK)
			throw DDSException("reigster_type() failed");

		DDS::Topic_var topic = dp->create_topic("Temperature", ts->get_type_name(), TOPIC_QOS_DEFAULT, 0, OpenDDS::DCPS::DEFAULT_STATUS_MASK);
		if (0 == topic)
			throw DDSException("create_topic() failed");

		DDS::Subscriber_var sub = dp->create_subscriber(SUBSCRIBER_QOS_DEFAULT, 0, OpenDDS::DCPS::DEFAULT_STATUS_MASK);
		if (0 == sub)
			throw DDSException("create_subscriber() failed");

		DDS::DataReaderQos dr_qos;
		sub->get_default_datareader_qos(dr_qos);

		rxsub::subject<rxsub::subject<Sensor::SensorData>> obsGroup;
		DDS::DataReaderListener_var listener(new DataReaderListenerImpl(obsGroup));

		DDS::DataReader_var dr = sub->create_datareader(topic, dr_qos, listener, OpenDDS::DCPS::DEFAULT_STATUS_MASK);
		if (0 == dr)
			throw DDSException("create_datareader() failed");

		obsGroup.get_observable()
			.subscribe(
			[](rxsub::subject<Sensor::SensorData> &obs) {
			obs.get_observable().subscribe(
				[](Sensor::SensorData &sd) {
				time_t date = sd.date;
				std::string cdate(ACE_OS::ctime(&date));
				cdate.erase(std::remove_if(cdate.begin(), cdate.end(), [](char c) { return c == 0x0A; }), cdate.end());
				std::cout << "[" << sd.sensorID << " " << cdate << " " << sd.reading << "]" << std::endl;
			},
				[]() { std::cout << "Completed" << std::endl; }
			);
		});

		// block until the publisher completes
		DDS::StatusCondition_var condition = dr->get_statuscondition();
		condition->set_enabled_statuses(DDS::SUBSCRIPTION_MATCHED_STATUS);

		DDS::WaitSet_var ws = new DDS::WaitSet;
		ws->attach_condition(condition);

		while (true) {
			DDS::SubscriptionMatchedStatus matches;
			if (dr->get_subscription_matched_status(matches) != DDS::RETCODE_OK)
				throw DDSException("get_subscription_matched_status failed");

			if (matches.current_count == 0 && matches.total_count > 0) {
				break;
			}

			DDS::ConditionSeq conditions;
			DDS::Duration_t timeout = { 60, 0 };
			if (ws->wait(conditions, timeout) != DDS::RETCODE_OK) 
				throw DDSException("wait failed");
		}

		ws->detach_condition(condition);

		dp->delete_contained_entities();
		dpf->delete_participant(dp);

		TheServiceParticipant->shutdown();

	}
	catch (const std::exception &e) {
		std::cout << "SensorSubscriber exception: " << e.what() << std::endl;
		return false;
	}
	catch (const CORBA::Exception &e) {
		e._tao_print_exception("SensorSubscriber exception: ");
		return false;
	}

	return 0;
}
