using System;
using System.Collections;
using System.Collections.Generic;
using System.Linq;
using System.Reactive.Disposables;
using System.Reactive.Linq;
using SensorEventLib;
using System.Reactive.Subjects;
using System.Collections.ObjectModel;
using System.Reactive.Concurrency;
using Microsoft.Reactive.Testing;
using System.Reactive;
using Microsoft.VisualStudio.TestTools.UnitTesting;


namespace reactive
{
    // if the class doesn't inherit from ReactiveTest as a base, have to spell out the Recorded<> entries
    [TestClass]
    public class NoBaseTests
    {
        [TestMethod]
        public void Produce42()
        {
            var scheduler = new TestScheduler();
            var xs = scheduler.CreateColdObservable(
                new Recorded<Notification<int>>(100, Notification.CreateOnNext(42)),
                new Recorded<Notification<int>>(200, Notification.CreateOnCompleted<int>())
            );

            var results = scheduler.CreateObserver<int>();

            xs.Subscribe(results);

            scheduler.Start();

            results.Messages.AssertEqual(
                new Recorded<Notification<int>>(100, Notification.CreateOnNext(42)),
                new Recorded<Notification<int>>(200, Notification.CreateOnCompleted<int>()));
        }
    }


    [TestClass]
    public class Tests : ReactiveTest
    {
        TestScheduler scheduler;
        ITestableObservable<int> xs, ys, zs;

        [TestInitialize]
        public void TestInit()
        {
            scheduler = new TestScheduler();
            xs = scheduler.CreateColdObservable(
                OnNext(10, 1),
                OnNext(20, 2),
                OnNext(40, 3),
                OnNext(41, 3),
                OnNext(60, 4),
                OnCompleted<int>(70)
            );
            ys = scheduler.CreateColdObservable(
                OnNext(5, 10),
                OnNext(15, 20),
                OnNext(45, 30),
                OnCompleted<int>(50)
            );
            zs = scheduler.CreateColdObservable(
                OnNext(10, 1),
                OnNext(20, 2),
                OnNext(30, 3),
                OnNext(40, 3),
                OnNext(50, 1),
                OnNext(60, 2),
                OnNext(70, 3),
                OnCompleted<int>(80)
            );
        }

        [TestMethod]
        public void Produce42()
        {
            var scheduler = new TestScheduler();
            var xs = scheduler.CreateColdObservable(
                OnNext(100, 42),
                OnCompleted<int>(200)
            );

            var results = scheduler.CreateObserver<int>();

            xs.Subscribe(results);

            scheduler.Start();

            results.Messages.AssertEqual(
                OnNext(100, 42),
                OnCompleted<int>(200));
        }

        [TestMethod]
        public void ProduceSensorEventArgs()
        {
            var scheduler = new TestScheduler();
            var xs = scheduler.CreateColdObservable(
                OnNext(100, new SensorEventArgs("Temp7", new DateTime(100), 42.0)),
                OnCompleted<SensorEventArgs>(200)
            );

            var results = scheduler.CreateObserver<SensorEventArgs>();

            xs.Subscribe(results);

            scheduler.Start();

            results.Messages.AssertEqual(
                OnNext(100, new SensorEventArgs("Temp7", new DateTime(100), 42.0)),
                OnCompleted<SensorEventArgs>(200));
        }

        [TestMethod]
        public void RealWorldTicks()
        {
            Assert.AreEqual(635241312000000000, new DateTime(2014, 1, 1, 0, 0, 0).Ticks);
        }

        [TestMethod]
        public void Half()
        {
            Assert.AreEqual(2, 4.Half());
        }


        [TestMethod]
        public void Merge()
        {
            var results = scheduler.CreateObserver<int>();
            xs.Merge(ys)
                .Subscribe(results);

            scheduler.Start();

            results.Messages.AssertEqual(
                OnNext(5, 10),
                OnNext(10, 1),
                OnNext(15, 20),
                OnNext(20, 2),
                OnNext(40, 3),
                OnNext(41, 3),
                OnNext(45, 30),
                OnNext(60, 4),
                OnCompleted<int>(70));
        }

        [TestMethod]
        public void Sample()
        {
            var results = scheduler.CreateObserver<int>();
            xs.Sample(TimeSpan.FromTicks(25), scheduler)  // have to specify the scheduler, otherwise the time interval applies to the default real time scheduler, giving incorrect test results
                .Subscribe(results);

            scheduler.Start();

            // most recent value at the end of the sample interval
            results.Messages.AssertEqual(
                OnNext(25, 2),
                OnNext(50, 3),
                OnNext(75, 4),
                OnCompleted<int>(75));
        }

        [TestMethod]
        public void Amb()
        {
            var results = scheduler.CreateObserver<int>();
            xs.Amb(ys)
                .Subscribe(results);

            scheduler.Start();

            results.Messages.AssertEqual(
                OnNext(5, 10),
                OnNext(15, 20),
                OnNext(45, 30),
                OnCompleted<int>(50));
        }

        [TestMethod]
        public void Where()
        {
            var results = scheduler.CreateObserver<int>();
            xs.Where(x => x > 2)
                .Subscribe(results);

            scheduler.Start();

            results.Messages.AssertEqual(
                OnNext(40, 3),
                OnNext(41, 3),
                OnNext(60, 4),
                OnCompleted<int>(70));
        }

        [TestMethod]
        public void Throttle()
        {
            var results = scheduler.CreateObserver<int>();
            xs.Throttle(TimeSpan.FromTicks(15), scheduler)
                .Subscribe(results);

            scheduler.Start();

            // skip 1 as not idle for 15 ticks before 2 appears
            // nothing within 15 ticks, so emit 2 15 ticks after it arrived at 20 (35)
            // skip first 3 since not idle, then emit 2nd 3 at 41+15=56 since idle for 15 ticks
            // emit last value at end of sequence
            results.Messages.AssertEqual(
                OnNext(35, 2),
                OnNext(56, 3),
                OnNext(70, 4),
                OnCompleted<int>(70));
        }

        // Distinct vs DistinctUntilChanged: http://www.introtorx.com/content/v1.0.10621.0/05_Filtering.html#Distinct
        [TestMethod]
        public void Distinct()
        {
            var results = scheduler.CreateObserver<int>();
            zs.Distinct()
                .Subscribe(results);

            scheduler.Start();

            results.Messages.AssertEqual(
                OnNext(10, 1),
                OnNext(20, 2),
                OnNext(30, 3),
                // drops every occurrence of 1,2,3 seen later
                OnCompleted<int>(80));
        }

        [TestMethod]
        public void DistinctUntilChanged()
        {
            var results = scheduler.CreateObserver<int>();
            zs.DistinctUntilChanged()
                .Subscribe(results);

            scheduler.Start();

            results.Messages.AssertEqual(
                OnNext(10, 1),
                OnNext(20, 2),
                OnNext(30, 3),
                // drops the second 3 only
                OnNext(50, 1),
                OnNext(60, 2),
                OnNext(70, 3),
                OnCompleted<int>(80));
        }

        [TestMethod]
        public void Concat()
        {
            var results = scheduler.CreateObserver<int>();
            xs.Concat(ys)
                .Subscribe(results);

            scheduler.Start();

            results.Messages.AssertEqual(
                // xs
                OnNext(10, 1),
                OnNext(20, 2),
                OnNext(40, 3),
                OnNext(41, 3),
                OnNext(60, 4),
                // ys
                OnNext(75, 10), // xs ends at 70, ys starts at 5
                OnNext(85, 20),
                OnNext(115, 30),
                // end
                OnCompleted<int>(120)
                );
        }

        // http://blogs.msdn.com/b/rxteam/archive/2012/06/14/testing-rx-queries-using-virtual-time-scheduling.aspx
        [TestMethod]
        public void GroupBy()
        {
            // as each group appears, add it to the groups list
            var groups = new List<Tuple<long, bool, ITestableObserver<int>>>();

            xs.GroupBy(x => x > 2)
                .Subscribe(g =>
                {
                    var observer = scheduler.CreateObserver<int>();
                    g.Subscribe(observer);
                    groups.Add(Tuple.Create(scheduler.Clock, g.Key, observer));
                });

            scheduler.Start();

            var assertGroup = new Action<int, long, bool, Recorded<Notification<int>>[]>(
                (index, clock, key, messages) =>
                {
                    var g = groups[index];
                    Assert.AreEqual(clock, g.Item1);
                    Assert.AreEqual(key, g.Item2);
                    g.Item3.Messages.AssertEqual(messages);
                });

            Assert.AreEqual(2, groups.Count);

            // at time 10, the "false" group appears
            assertGroup(0, 10, false, new[] {
                OnNext(10, 1),
                OnNext(20, 2),
                OnCompleted<int>(70)
            });

            // at time 40 the "true" group appears
            assertGroup(1, 40, true, new[] {
                OnNext(40, 3),
                OnNext(41, 3),
                OnNext(60, 4),
                OnCompleted<int>(70)
            });
        }

        [TestMethod]
        public void Window()
        {
            var groups = new List<Tuple<long, ITestableObserver<int>>>();

            xs.Window(TimeSpan.FromTicks(25), scheduler)
                .Subscribe(g =>
                {
                    var observer = scheduler.CreateObserver<int>();
                    g.Subscribe(observer);
                    groups.Add(Tuple.Create(scheduler.Clock, observer));
                });

            scheduler.Start();

            var assertGroup = new Action<int, long, Recorded<Notification<int>>[]>(
                (index, clock, messages) =>
                {
                    var g = groups[index];
                    Assert.AreEqual(clock, g.Item1);
                    g.Item2.Messages.AssertEqual(messages);
                });

            Assert.AreEqual(3, groups.Count);

            // time 0
            assertGroup(0, 0, new[] {
                OnNext(10, 1),
                OnNext(20, 2),
                OnCompleted<int>(25)  // end of window
            });

            // time 25
            assertGroup(1, 25, new[] {
                OnNext(40, 3),
                OnNext(41, 3),
                OnCompleted<int>(50)  // end of window
            });

            // time 50
            assertGroup(2, 50, new[] {
                OnNext(60, 4),
                OnCompleted<int>(70) // end of sequence is before the end of the window
            });

        }

        // buffer vs. window: http://www.introtorx.com/uat/content/v1.0.10621.0/17_SequencesOfCoincidence.html
        [TestMethod]
        public void Buffer_Count()
        {
            var groups = new List<Tuple<long, ITestableObserver<int>>>();

            // can buffer on a time interval, but use a count for the example
            xs.Buffer(3)
                .Subscribe(g =>
                {
                    var observer = scheduler.CreateObserver<int>();
                    g.Subscribe(observer);
                    groups.Add(Tuple.Create(scheduler.Clock, observer));
                });

            scheduler.Start();

            var assertGroup = new Action<int, long, Recorded<Notification<int>>[]>(
                (index, clock, messages) =>
                {
                    var g = groups[index];
                    Assert.AreEqual(clock, g.Item1);
                    g.Item2.Messages.AssertEqual(messages);
                });

            Assert.AreEqual(2, groups.Count);

            assertGroup(0, 40, new[] {
                OnNext(40, 1),
                OnNext(40, 2),
                OnNext(40, 3),
                OnCompleted<int>(40)  // count of 3 elements reached
            });

            assertGroup(1, 70, new[] {
                OnNext(70, 3),
                OnNext(70, 4),
                OnCompleted<int>(70)  // end of sequence before 3 elements reached, so buffer has only 2 elements in it
            });
        }

        [TestMethod]
        public void Buffer_Time()
        {
            var groups = new List<Tuple<long, ITestableObserver<int>>>();

            // can buffer on a time interval, but use a count for the example
            xs.Buffer(TimeSpan.FromTicks(25), scheduler)
                .Subscribe(g =>
                {
                    var observer = scheduler.CreateObserver<int>();
                    g.Subscribe(observer);
                    groups.Add(Tuple.Create(scheduler.Clock, observer));
                });

            scheduler.Start();

            var assertGroup = new Action<int, long, Recorded<Notification<int>>[]>(
                (index, clock, messages) =>
                {
                    var g = groups[index];
                    Assert.AreEqual(clock, g.Item1);
                    g.Item2.Messages.AssertEqual(messages);
                });

            Assert.AreEqual(3, groups.Count);

            // time is at the end of the buffer interval (or end of sequence)
            assertGroup(0, 25, new[] {
                OnNext(25, 1),
                OnNext(25, 2),
                OnCompleted<int>(25)  // end of window
            });

            assertGroup(1, 50, new[] {
                OnNext(50, 3),
                OnNext(50, 3),
                OnCompleted<int>(50)  // end of window
            });

            assertGroup(2, 70, new[] {
                OnNext(70, 4),
                OnCompleted<int>(70) // end of sequence is before the end of the window
            });
        }



        [TestMethod]
        public void Zip()
        {
            var results = scheduler.CreateObserver<int>();
            xs.Zip(ys, (x, y) => x + y)
                .Subscribe(results);

            scheduler.Start();

            results.Messages.AssertEqual(
                OnNext(10, 11),
                OnNext(20, 22),
                OnNext(45, 33),
                OnCompleted<int>(60));
        }


        [TestMethod]
        public void ToCommonAggregates()
        {
            var results = scheduler.CreateObserver<StatInfoItem<string>>();
            var obs = ys  // IObservable<int>
                .Timestamp(scheduler)  // now a IObservable<Timestamped<int>>
                .Select(i => new SensorEventArgs("Temp7", i.Timestamp.DateTime, i.Value)) // IObservable<SensorEventArgs>
                .ToCommonAggregates(i => i.Reading, i => i.SensorID);  // IObservable<StatInfoItem<string>>

            obs.Subscribe(results);

            scheduler.Start();

            results.Messages.AssertEqual(
                OnNext(15, new StatInfoItem<string>()
                {
                    Item = "Temp7",
                    Sum = 30.0,
                    Count = 2,
                    Mean = 15.0,
                    M2 = 50.0,
                    StdDev = 5.0,
                    Min = 0.0,
                    Max = 20.0,
                }),
                OnNext(45, new StatInfoItem<string>()
                {
                    Item = "Temp7",
                    Sum = 60.0,
                    Count = 3,
                    Mean = 20.0,
                    M2 = 200.0,
                    StdDev = 8.16496580927726,
                    Min = 0.0,
                    Max = 30.0,
                }),
                OnCompleted<StatInfoItem<string>>(50)
                );
        }


        [TestMethod]
        public void GroupJoin()
        {
            // http://rxwiki.wikidot.com/101samples#toc45

            /*
            var l = Observable.Generate(
                0,
                i => i < 40,
                i => i + 1,
                i => new string[] { DateTime.Now.ToString("G"), "Batch" + i },
                i => TimeSpan.FromSeconds(1)
             );

            var r = Observable.Generate(
                0,
                i => i < 20,
                i => i + 1,
                i => new string[] { DateTime.Now.ToString("G"), "Production=" + i },
                i => TimeSpan.FromSeconds(2)
             );
            */
            
            var leftList = new List<string[]>();
            leftList.Add(new string[] { "2013-01-01 02:00:00", "Batch1" });
            leftList.Add(new string[] { "2013-01-01 03:00:00", "Batch2" });
            leftList.Add(new string[] { "2013-01-01 04:00:00", "Batch3" });

            var rightList = new List<string[]>();
            rightList.Add(new string[] { "2013-01-01 01:00:00", "Production=2" });
            rightList.Add(new string[] { "2013-01-01 02:00:00", "Production=0" });

            rightList.Add(new string[] { "2013-01-01 02:00:01", "Production=0" });
            rightList.Add(new string[] { "2013-01-01 02:00:02", "Production=0" });
            rightList.Add(new string[] { "2013-01-01 02:00:03", "Production=0" });

            rightList.Add(new string[] { "2013-01-01 03:00:00", "Production=3" });
             
            var l = leftList.ToObservable();
            var r = rightList.ToObservable(); 
            
            var q = l.GroupJoin(r,
                _ => Observable.Never<Unit>(), // windows from each left event going on forever
                _ => Observable.Never<Unit>(), // windows from each right event going on forever
                (left, obsOfRight) => Tuple.Create(left, obsOfRight)); // create tuple of left event with observable of right events

            var messages = new List<string>();

            // e is a tuple with two items, left and obsOfRight
            using (q.Subscribe(e =>
            {
                var xs = e.Item2;
                xs.Where(
                 x => x[0] == e.Item1[0]) // filter only when datetime matches
                 .Subscribe(
                 v =>
                 {
                     messages.Add(string.Format(
                        string.Format("{0},{1} and {2},{3} occur at the same time",
                        e.Item1[0],
                        e.Item1[1],
                        v[0],
                        v[1]
                     )));
                 });
            }))
            {

                Assert.AreEqual(2, messages.Count);
                Assert.AreEqual(
                    "2013-01-01 02:00:00,Batch1 and " +
                    "2013-01-01 02:00:00,Production=0 " +
                    "occur at the same time", messages[0]);
                Assert.AreEqual(
                    "2013-01-01 03:00:00,Batch2 and " +
                    "2013-01-01 03:00:00,Production=3 " +
                    "occur at the same time", messages[1]);          
            }
        }


        [TestMethod]
        public void CountSumMinMaxAverageAggregate()
        {
            var o = new List<int>() { 3, 10, 8 }.ToObservable();
            o.Count().Subscribe(e => Assert.AreEqual(3, e));
            o.Sum().Subscribe(e => Assert.AreEqual(21, e));
            o.Min().Subscribe(e => Assert.AreEqual(3, e));
            o.Max().Subscribe(e => Assert.AreEqual(10, e));
            o.Average().Subscribe(e => Assert.AreEqual(7, e));
            o.Aggregate(6, (acc, i) => acc + i*2)
                .Subscribe(e => Assert.AreEqual(48, e));
        }

    }





    class Program
    {
        static void BasicObservable()
        {
            IObservable<int> o = Observable.Create<int>(observer =>
            {
                observer.OnNext(42);
                observer.OnCompleted();
                return Disposable.Empty; // use Create(Action) if it needs to take an action when disposed
            });

            var subscription = o.Subscribe(
                onNext: x => { Console.WriteLine("Next: " + x); },
                onError: ex => { Console.WriteLine("Exception: " + ex); },
                onCompleted: () => { Console.WriteLine("Done"); }
                );
        }

        static void TimedObservable()
        {
            // one count per second, increasing values
            var o = Observable.Generate(
                0,
                i => i < 10,
                i => i + 1,
                i => i,
                i => TimeSpan.FromSeconds(1)
             );

            // ignore OnError, OnCompleted - don't provide methods to call when they are invoked
            using (var subscription = o.Subscribe(Console.WriteLine))
            {
                Console.WriteLine("Press ENTER to stop");
                Console.ReadLine();
            }
        }


        static void SensorPublisher()
        {
            // Simulate what SensorPublisher does

            var o = Observable.Generate(
                0,
                i => i < 20,
                i => i + 1,
                i => new SensorEventLib.SensorEventArgs("Temp7", DateTime.Now, i % 10),
                i => TimeSpan.FromSeconds(2)
             );

            using (o.Subscribe(Console.WriteLine))
            {
                Console.WriteLine("Press ENTER to stop");
                Console.ReadLine();
            }
        }


        static void TimeMerge()
        {
            // Correlate sample streams based on time index

            var oa = Observable.Generate(
                0,
                i => i < 40,
                i => i + 1,
                i => new SensorEventArgs("A", DateTime.Now, i % 20),
                i => TimeSpan.FromSeconds(1)
             );

            var ob = Observable.Generate(
                0,
                i => i < 20,
                i => i + 1,
                i => new SensorEventArgs("B", DateTime.Now, i % 10),
                i => TimeSpan.FromSeconds(2)
             );

            var res = oa.TimeMerge(ob, TimeSpan.FromSeconds(30));

            using (res.Subscribe(Console.WriteLine))
            {
                Console.WriteLine("Press ENTER to stop");
                Console.ReadLine();
            }
        }



        static void Main(string[] args)
        {
            // TimeMerge();
            new Tests().GroupJoin();

        }
    }

    class TimeMerger
    {
        Dictionary<DateTime, SensorEventArgs> _ad = new Dictionary<DateTime, SensorEventArgs>();
        Dictionary<DateTime, SensorEventArgs> _bd = new Dictionary<DateTime, SensorEventArgs>();
        Subject<Tuple<SensorEventArgs, SensorEventArgs>> _sbj = new Subject<Tuple<SensorEventArgs, SensorEventArgs>>();
        TimeSpan _span;
        private Object _filterLock = new Object();

        delegate void FilterDelegate(SensorEventArgs a, Dictionary<DateTime, SensorEventArgs> l);

        public TimeMerger(TimeSpan span)
        {
            _span = span;
        }

        void Filter(SensorEventArgs a, Dictionary<DateTime, SensorEventArgs> l)
        {
            // drop anything from al and bl that is older than now-span
            // match elements of al to bl and if match, pull out and subj.OnNext() them
            lock (_filterLock)
            {
                // want to synchronize to the second, so reduce the precision for easier comparison
                l.Add(new DateTime(a.Date.Year, a.Date.Month, a.Date.Day, a.Date.Hour, a.Date.Minute, a.Date.Second), a);

                DateTime oldest = DateTime.Now - _span;
                _ad = _ad.Where(e => e.Key >= oldest).ToDictionary(x => x.Key, x => x.Value);
                _bd = _bd.Where(e => e.Key >= oldest).ToDictionary(x => x.Key, x => x.Value);

                // without ToList() get InvalidOperationException - collection was modified
                var matches = _ad.Keys.Intersect(_bd.Keys).ToList();

                foreach (var m in matches)
                {
                    _sbj.OnNext(Tuple.Create(_ad[m], _bd[m]));
                    _ad.Remove(m);
                    _bd.Remove(m);
                }
            }
        }

        public IObservable<Tuple<SensorEventArgs, SensorEventArgs>> TimeMerge(IObservable<SensorEventArgs> oa, IObservable<SensorEventArgs> ob)
        {
            oa.Subscribe(a => (new FilterDelegate(this.Filter))(a, _ad));
            ob.Subscribe(b => (new FilterDelegate(this.Filter))(b, _bd));
            return _sbj;
        }
    }






    // http://weblogs.asp.net/sweinstein/streaming-olap-with-the-reactive-extensions-rx-for-net-part-1
    public class StatInfoItem<T> : IEquatable<StatInfoItem<T>>
    {
        public T Item { get; set; }
        public double Sum { get; set; }
        public int Count { get; set; }
        public double Mean { get; set; }
        public double M2 { get; set; }
        public double StdDev { get; set; }
        public double Min { get; set; }
        public double Max { get; set; }

        public override string ToString()
        {
            return "[" + Item + ": Sum=" + Sum + ", Count=" + Count + ", Mean=" + Mean + ", M2=" + M2 + ", StdDev=" + StdDev + ", Min=" + Min + ", Max=" + Max + "]";
        }

        // since results.Messages.AssertEqual() uses the default comparator, obtained from IEquatable<>, implement this
        public bool Equals(StatInfoItem<T> other)
        {
            const double eps = 0.00001;
            return Item.Equals(other.Item) &&
                Count.Equals(other.Count) &&
                NearlyEqual(Sum, other.Sum, eps) &&
                NearlyEqual(Mean, other.Mean, eps) &&
                NearlyEqual(M2, other.M2, eps) &&
                NearlyEqual(StdDev, other.StdDev, eps) &&
                NearlyEqual(Min, other.Min, eps) &&
                NearlyEqual(Max, other.Max, eps);
        }

        // http://stackoverflow.com/questions/3874627/floating-point-comparison-functions-for-c-sharp
        public bool NearlyEqual(double a, double b, double epsilon)
        {
            double absA = Math.Abs(a);
            double absB = Math.Abs(b);
            double diff = Math.Abs(a - b);

            if (a == b)
            {
                // shortcut, handles infinities
                return true;
            }
            else if (a == 0 || b == 0 || diff < Double.MinValue)
            {
                // a or b is zero or both are extremely close to it
                // relative error is less meaningful here
                return diff < (epsilon * Double.MinValue);
            }
            else
            {
                // use relative error
                return diff / (absA + absB) < epsilon;
            }
        }
    }


    static class Extensions
    {
        public static IObservable<Tuple<SensorEventArgs, SensorEventArgs>> TimeMerge(this IObservable<SensorEventArgs> oa, IObservable<SensorEventArgs> ob, TimeSpan span)
        {
            return new TimeMerger(span).TimeMerge(oa, ob);
        }

        public static int Half(this int source) {
            return source / 2;
        } 

        // could, say, include another parameter which would reset values when a count or time duration was reached
        public static IObservable<StatInfoItem<T>> ToCommonAggregates<T, TSrc>(
           this IObservable<TSrc> source,
           Func<TSrc, double> dataSelector,
           Func<TSrc, T> itemSelector)
        {
            return source.Scan(new StatInfoItem<T>(), (cur, next) =>
            {
                double data = dataSelector(next);
                T itemp = itemSelector(next);
                var n = cur.Count + 1;
                var delta = data - cur.Mean;
                var meanp = cur.Mean + delta / n;
                var m2 = cur.M2 + delta * (data - meanp);
                var stdDevp = Math.Sqrt(m2 / n);
                return new StatInfoItem<T>()
                {
                    Item = itemp,
                    Sum = data + cur.Sum,
                    Count = n,
                    Mean = meanp,
                    M2 = m2,
                    StdDev = stdDevp,
                    Min = Math.Min(data, cur.Min),
                    Max = Math.Max(data, cur.Max),
                };
            })
            .Skip(1); // need a seed, but don't want to include seed value in the output
        }

    }
}
