#include "rx.hpp"
#include <string>

namespace rx = rxcpp;
namespace rxu = rxcpp::util;
namespace rxsc = rxcpp::schedulers;

// see RxCPP\Rx\v2\examples\println\main.cpp
void Test1() {
    auto get_names = [](){return rx::observable<>::from<std::string>(
        "Matthew",
        "Aaron"
        ); };

    std::cout << "===== println stream of std::string =====" << std::endl;
    auto hello_str = [&](){return get_names().map([](std::string n){
        return "Hello, " + n + "!";
    }).as_dynamic(); };

    hello_str().subscribe(rxu::println(std::cout));

    std::cout << "===== println stream of std::tuple =====" << std::endl;
    auto hello_tpl = [&](){return get_names().map([](std::string n){
        return std::make_tuple("Hello, ", n, "! (", n.size(), ")");
    }).as_dynamic(); };

    hello_tpl().subscribe(rxu::println(std::cout));

    hello_tpl().subscribe(rxu::print_followed_by(std::cout, " and "), rxu::endline(std::cout));
}


// see RxCPP\Rx\v2\test\sources\interval.cpp
void Test2() {
    using namespace std::chrono;
    typedef steady_clock clock;

    int c = 0;
    auto sc = rxsc::make_current_thread();
    auto w = sc.create_worker();
    auto start = w.now() + seconds(2);
    auto period = seconds(1);
    w.schedule_periodically(start, period,
        [=, &c](rxsc::schedulable scbl){
        auto nsDelta = duration_cast<milliseconds>(scbl.now() - (start + (period * c)));
        ++c;
        std::cout << "schedule_periodically          : period " << c
            << ", " << nsDelta.count() << "ms delta from target time" << std::endl;
        if (c == 5) { scbl.unsubscribe(); }
    });
}


// see http://ledentsov.de/2014/06/22/rxcpp-cplusplus-background-ticker-easy/
// but doesn't work with current syntax
void Test3() {
    //auto scheduler = std::make_shared<rxcpp::EventLoopScheduler>();
    //auto ticker = rxcpp::Interval(std::chrono::milliseconds(250), scheduler);

    auto sc = rxsc::make_current_thread();
    auto so = rx::synchronize_in_one_worker(sc);
    auto ticker =
        rx::observable<>::interval(sc.now(), std::chrono::milliseconds(250), so);

    ticker
        .filter([](int x) {
            return x % 2 == 0;
        })
        .subscribe([](int val) {
        std::cout << "tick " << val << std::endl;
    });

    std::cout << "starting to tick" << std::endl;
}


void Test4() {
    //auto scheduler = std::make_shared<rxcpp::EventLoopScheduler>();
    //auto ticker = rxcpp::Interval(std::chrono::milliseconds(250), scheduler);

    auto sc = rxsc::make_current_thread();
    auto so = rx::synchronize_in_one_worker(sc);
    auto ticker =
        rx::observable<>::interval(sc.now(), std::chrono::milliseconds(250), so);

    ticker
        .map([](int x) {
            return "Fred" + std::to_string(x);
        })
        //.as_dynamic()  // forget type to workaround lambda deduction bug on msvc 2013
        .subscribe([](const std::string &val) {
        std::cout << "tick " << val << std::endl;
    });

    std::cout << "starting to tick" << std::endl;
}


void Test5() {
    using namespace std::chrono;
    typedef steady_clock clock;

    int c;
    int n = 1;

    c = 0;
    auto start = clock::now();
    auto o = rx::make_observer<int>(
        [&c](int){++c; },
        [](std::exception_ptr){abort(); });
    for (int i = 0; i < 10; i++) {
        o.on_next(i);
    }
    o.on_completed();
    auto finish = clock::now();
    auto msElapsed = duration_cast<milliseconds>(finish - start);
    std::cout << "loop -> observer    : " << n << " subscribed, " << c << " on_next calls, " << msElapsed.count() << "ms elapsed " << std::endl;
}


typedef std::chrono::system_clock clk;
typedef std::chrono::time_point<clk> tp;

struct SensorData {
    std::string _sensorID;
    tp _date;
    double _reading;

    SensorData(const std::string &sensorID, const tp &date, const double reading) :
        _sensorID(sensorID), _date(date), _reading(reading) {}
};

void DDS1() {
	auto sc = rxcpp::schedulers::make_current_thread();
	auto so = rxcpp::synchronize_in_one_worker(sc);
    auto o =
        rxcpp::observable<>::interval(sc.now(), std::chrono::seconds(2), so)
		.take(20)
		.map([](int i) {
        return SensorData("Temp7", clk::now(), i%10);
    });

    o.subscribe([](SensorData &sd) {
		std::time_t date = std::chrono::system_clock::to_time_t(sd._date);
		std::string cdate(std::ctime(&date));
		cdate.erase(std::remove_if(cdate.begin(), cdate.end(), [](char c) { return c == 0x0A; }), cdate.end());
        std::cout << "[" << sd._sensorID << " " << cdate << " " << sd._reading << "]" << std::endl;
    });

}

int main() {
    DDS1();
    
    return 0;
}