package Parser

import scala.util.parsing.combinator._
import scala.language.implicitConversions


case class Name(name: String) {
  override def toString() = s"""%NAME\n"$name"\n"""
}

case class Address(domain: Int, subnet: Int, node: Int) {
  override def toString() = s"%ADDRESS\n$domain.$subnet.$node\n"
}

case class HardwareID(id: String) {
  override def toString() = s"%HWID\n$id\n"
}

case class DevInfo(name: Name, address: Address, hardwareID: HardwareID) {
  override def toString() = s"%DEVINFO<$name$address$hardwareID>"
}

trait DaqRateType {}

case class DaqRate_Event() extends DaqRateType {
  override def toString() = "E"
}

case class DaqRate_Periodic(interval: Int) extends DaqRateType {
  override def toString() = s"P,$interval"
}

case class DaqRate(daqRate: DaqRateType) {
  override def toString() = s"%DR\n$daqRate\n"
}

case class Range(min: Double, max: Double) {
  override def toString() = s"%RANGE\n$min,$max\n"
}

case class SAType(saType: Int) {
  override def toString() = s"%TYPE\n$saType\n"
}

case class Enabled(enabled: Boolean) {
  override def toString() = "%EN\n" + (if (enabled) "1" else "0") + "\n"
}

case class Diagnostic(diagnostic: String) {
  override def toString() = s"%DIAGSTART\n$diagnostic%DIAGEND\n"
}

case class Sensor(index: Int, name: Name, daqRate: Option[DaqRate], range: Option[Range], saType: Option[SAType], 
    enabled: Option[Enabled], diagnostic: Option[Diagnostic]) {
  override def toString() = s"%SENSOR$index<$name" +
    daqRate.map(_.toString).getOrElse("") +
    range.map(_.toString).getOrElse("") +
    saType.map(_.toString).getOrElse("") +
    enabled.map(_.toString).getOrElse("") +
    diagnostic.map(_.toString).getOrElse("") + ">"
}

case class Actuator(index: Int, name: Name, range: Option[Range], saType: Option[SAType], 
    enabled: Option[Enabled], diagnostic: Option[Diagnostic]) {
  override def toString() = s"%ACTUATOR$index<$name" +
    range.map(_.toString).getOrElse("") +
    saType.map(_.toString).getOrElse("") +
    enabled.map(_.toString).getOrElse("") +
    diagnostic.map(_.toString).getOrElse("") + ">"
}

case class Device(devInfo: DevInfo, sensors: List[Sensor], actuators: List[Actuator]) {
  override def toString() = "[" + devInfo.toString + sensors.mkString + actuators.mkString + "]"
}

object ConfigParser extends RegexParsers {
  override val skipWhitespace = false // otherwise \n is absorbed and not parsed

  def nl: Parser[String] = "\n"  // otherwise "String"
  def int = "[-+]?[0-9]+".r ^^ { _.toInt }
  def bool = ("0" | "1") ^^ { _.toInt != 0 }
  def double = """[-+]?[0-9]*\.?[0-9]+([eE][-+]?[0-9]+)?""".r ^^ { _.toDouble }
  def qstring = "\"" ~> ("\"\"" | "[^\"]".r).* <~ "\"" ^^ { _.mkString /* outer quotes are removed */ }
  
  
  def name = "%NAME" ~ nl ~ qstring ~ nl ^^ {
    case _ ~ _ ~ name ~ _ => Name(name)
  }
  
  def address = "%ADDRESS" ~ nl ~ int ~ "." ~ int ~ "." ~ int ~ nl ^^ {
    case _ ~ _ ~ domain ~ _ ~ subnet ~ _ ~ node ~ _ => Address(domain, subnet, node)
  }
  
  def hardwareID = "%HWID" ~ nl ~ "[0-9A-F]{8}".r ~ nl ^^ {
    case _ ~ _ ~ id ~ _ => HardwareID(id)
  }
  
  def devInfo = "%DEVINFO" ~ "<" ~ name ~ address ~ hardwareID ~ ">" ^^ {
    case _ ~ _ ~ name ~ address ~ hardware_id ~ _ => DevInfo(name, address, hardware_id)
  }
  
  def daqRate_Event = "E" ^^ {
    case _ => DaqRate_Event()
  }
  
  def daqRate_Periodic = "P," ~ int ^^ {
    case _ ~ interval => DaqRate_Periodic(interval)
  }
  
  def daqRate = "%DR" ~ nl ~ (daqRate_Event | daqRate_Periodic) ~ nl ^^ {
    case _ ~ _~ daqRate ~ _ => DaqRate(daqRate)
  }
  
  def range = "%RANGE" ~ nl ~ double ~ "," ~ double ~ nl ^^ {
    case _ ~ _ ~ min ~ _ ~ max ~ _ => Range(min, max)
  }
  
  def saType = "%TYPE" ~ nl ~ int ~ nl ^^ {
    case _ ~ _ ~ saType ~ _ => SAType(saType)
  }
  
  def enabled = "%EN" ~ nl ~ bool ~ nl ^^ {
    case _ ~ _ ~ enabled ~ _ => Enabled(enabled)
  }
  
   // should begin with %DIAG, take any chars that follow until another %DIAG or end of string
   def diagnostic = new Parser[Diagnostic] {
     def apply(in: Input) = {
       val source = in.source
       val offset = in.offset
       val start = handleWhiteSpace(source, offset)
       val secStart = "%DIAGSTART\n"
       val secEnd = "%DIAGEND\n"
       val text = source.subSequence(start, source.length).toString
       
       val iStart = text.indexOf(secStart)  // -1 if no match
       if (iStart>=0) {
         val contents = text.drop(iStart + secStart.length)
         val iEnd = contents.indexOf(secEnd)
         if (iEnd == -1) 
           Success(Diagnostic(contents), in.drop(start + secStart.length + contents.length - offset))
         else {
           val strippedContents = contents.take(iEnd)
           Success(Diagnostic(strippedContents), in.drop(start + secStart.length + strippedContents.length + secEnd.length - offset))
         }
       }
       else
         Failure(s"$secStart not found", in.drop(start - offset))
     }
 }
   
   def sensor = "%SENSOR" ~ int ~ "<" ~ name ~ daqRate.? ~ range.? ~ saType.? ~ enabled.? ~ diagnostic.? ~ ">" ^^ {
     case _ ~ index ~ _ ~ name ~ daqRate ~ range ~ saType ~ enabled ~ diagnostic ~ _ => 
       Sensor(index, name, daqRate, range, saType, enabled, diagnostic)
   }
   
    def actuator = "%ACTUATOR" ~ int ~ "<" ~ name ~ range.? ~ saType.? ~ enabled.? ~ diagnostic.? ~ ">" ^^ {
     case _ ~ index ~ _ ~ name ~ range ~ saType ~ enabled ~ diagnostic ~ _ => 
       Actuator(index, name, range, saType, enabled, diagnostic)
   }  
    
    def device = "[" ~> devInfo ~ sensor.* ~ actuator.* <~ "]" ^^ {
      case devInfo ~ sensors ~ actuators  => Device(devInfo, sensors, actuators)
    }
  
  def parseConfig(s: String): Option[Device] = {
      ConfigParser.parse(ConfigParser.device, s) match {
       case ConfigParser.Success(matched,_) => Some(matched)
       case ConfigParser.Failure(msg,_) => { println("Parse failure of " + s + " due to: " + msg); None }
       case ConfigParser.Error(msg,_) => { println("Parse error of " + s + " due to: " + msg); None }
    }
  }
}
 

