import org.specs2.mutable._
import scala.util.parsing.combinator.RegexParsers
import org.specs2.matcher.ParserMatchers

import Parser._

class ConfigParserSpec extends Specification with ParserMatchers  {
  val parsers = ConfigParser
  
  import ConfigParser._

 
  "qstring" should {
    "recognize \"fred\"" in {
      qstring must succeedOn("\"fred\"").withResult("fred")
    }
    
    "not recognize fred\"" in {
      qstring must not succeedOn("fred\"")
    }
    
    "not recognize \"fred" in {
      qstring must failOn("\"fred")
    }
  }
  

  "name" should {
    "recognize '%NAME\n\"fred\"\n'" in {
      name must succeedOn("%NAME\n\"fred\"\n").withResult(Name("fred"))
    }
    "encode Name(\"fred\") correctly" in {
      Name("fred").toString === "%NAME\n\"fred\"\n"
    }
  }

  
  def roundtrip[A](e: A, parser: ConfigParser.Parser[A]) = {
    "roundtrip " + e.toString.replace("\n", "\\n") in {
      parser must succeedOn(e.toString).withResult(e)
    }
  }
  
  def succeed[A](raw: String, e: A, parser: ConfigParser.Parser[A]) = {
    "recognize " + raw.replace("\n", "\\n") in {
      parser must succeedOn(raw).withResult(e)
    }
  }
  
  def fail[A](raw: String, parser: ConfigParser.Parser[A]) = {
    "fail to parse " + raw.replace("\n", "\\n") in {
      parser must not succeedOn(raw)
    }
  }
 
  roundtrip(Name("fred"), ConfigParser.name)

  roundtrip(Address(1,2,3), ConfigParser.address)
  
  roundtrip(HardwareID("0123ABCD"), ConfigParser.hardwareID)
  fail("%HWID\n0123ABCDE\n", ConfigParser.hardwareID) // too long
  fail("%HWID\n0123ABC\n", ConfigParser.hardwareID) // too short
  fail("%HWID\n0123ABCx\n", ConfigParser.hardwareID) // not all hex chars
  
  roundtrip(DevInfo(Name("fred"), Address(1,2,3), HardwareID("0123ABCD")), ConfigParser.devInfo)
  
  roundtrip(DaqRate(DaqRate_Event()), ConfigParser.daqRate)
  roundtrip(DaqRate(DaqRate_Periodic(3)), ConfigParser.daqRate)
  
  roundtrip(Range(-20.1, 212.9), ConfigParser.range)
  
  roundtrip(SAType(1), ConfigParser.saType)
  
  roundtrip(Enabled(true), ConfigParser.enabled)
  
  roundtrip(Diagnostic("a diagnostic \"nested string\""), ConfigParser.diagnostic)
  succeed("%DIAGSTART\ndiagnostic", Diagnostic("diagnostic"), ConfigParser.diagnostic)
  
  roundtrip(Sensor(1, Name("Temperature"), Some(DaqRate(DaqRate_Periodic(60))), 
      Some(Range(-20, 120)), Some(SAType(1)), Some(Enabled(true)), None), ConfigParser.sensor)
  roundtrip(Sensor(1, Name("Temperature"), None, None, None, None, None), ConfigParser.sensor) 
  
  roundtrip(Actuator(1, Name("Dimmer"),  
      Some(Range(0, 100)), Some(SAType(7)), Some(Enabled(true)), None), ConfigParser.actuator)  
      
  roundtrip(Sensor(1, Name("Temperature"), Some(DaqRate(DaqRate_Periodic(60))), 
            Some(Range(-20, 120)), Some(SAType(1)), Some(Enabled(true)), None), 
            ConfigParser.sensor)
  roundtrip(Sensor(2, Name("Humidity"), Some(DaqRate(DaqRate_Periodic(300))), 
            Some(Range(0, 100)), Some(SAType(2)), Some(Enabled(true)), 
            Some(Diagnostic("I'm broken"))), 
            ConfigParser.sensor)
  roundtrip(Sensor(3, Name("IsRaining"), Some(DaqRate(DaqRate_Event())), 
            Some(Range(0, 1)), Some(SAType(3)), None, None),
            ConfigParser.sensor)
  roundtrip(Actuator(1, Name("Alarm"),  
            Some(Range(0, 1)), Some(SAType(1)), Some(Enabled(true)), None),
            ConfigParser.actuator)
            
  roundtrip(Device(DevInfo(Name("Weather Station"), Address(1,2,3), HardwareID("0123ABCD")),
      List(), List()), ConfigParser.device)
      
  roundtrip(Device(
      DevInfo(Name("Weather Station"), Address(1,2,3), HardwareID("0123ABCD")),
        List(
           Sensor(1, Name("Temperature"), Some(DaqRate(DaqRate_Periodic(60))), 
             Some(Range(-20, 120)), Some(SAType(1)), Some(Enabled(true)), 
               Some(Diagnostic("I'm broken"))),     
           Sensor(2, Name("Humidity"), Some(DaqRate(DaqRate_Periodic(300))), 
             Some(Range(0, 100)), Some(SAType(2)), Some(Enabled(true)), 
               None),
           Sensor(3, Name("IsRaining"), Some(DaqRate(DaqRate_Event())), 
             Some(Range(0, 1)), Some(SAType(3)), None, None)
             ),
         List(
           Actuator(1, Name("Alarm"),  
             Some(Range(0, 1)), Some(SAType(1)), Some(Enabled(true)), None))),
      ConfigParser.device)
      
      
}
